// ============================================================================
// File:               $File$
//
// Project:            
//
// Purpose:            
//
// Authors:            Harald Boegeholz (original C++ code)
//                     Rammi (port to Java and more)
//
// Copyright Notice:   (c) 2008  Harald Boegehoz/c't
//                     (c) 2008  Rammi (rammi@caff.de)
//                     This code is in the public domain.
//                     Use at own risk.
//                     No guarantees given.
//
// Latest change:      $Date$
//
// History:	       $Log$
//=============================================================================
package de.caff.asteroid;

import java.awt.*;
import java.io.IOException;
import java.util.*;
import java.util.List;

/**
 *  The info describing one game frame constructed from a received datagram.
 *
 *  This class is part of a solution for a
 *  <a href="http://www.heise.de/ct/creativ/08/02/details/">competition by the German computer magazine c't</a>.
 */
public class FrameInfo
        implements GameData
{
  static final int MAME_DATAGRAM_SIZE  = VECTORRAM_SIZE + 2;
  private static final int ID_BYTE = VECTORRAM_SIZE;
  private static final int PING_BYTE = ID_BYTE+1;

  private static final int MASK_OPCODE = 0xF0;
  private static final int OPCODE_LABS = 0xA0;
  private static final int OPCODE_HALT = 0xB0;
  private static final int OPCODE_JSRL = 0xC0;
  private static final int OPCODE_RTSL = 0xD0;
  private static final int OPCODE_JMPL = 0xE0;
  private static final int OPCODE_SVEC = 0xF0;
  private static final int OPCODE_SHIP = 0x60;

  private static final int SUB_ASTEROID_TYPE1 = 0x8F3;
  private static final int SUB_ASTEROID_TYPE2 = 0x8FF;
  private static final int SUB_ASTEROID_TYPE3 = 0x90D;
  private static final int SUB_ASTEROID_TYPE4 = 0x91A;
  private static final int SUB_UFO            = 0x929;

  /** Frame ID as sent by MAME. */
  private final byte id;
  /** Ping ID as sent by MAME. */
  private final byte ping;
  /** The ufo (if any). */
  private Ufo ufo;
  /** The list of asteroids. */
  private List<Asteroid> asteroids = new LinkedList<Asteroid>();
  /** The spaceship (if any). */
  private SpaceShip spaceShip;
  /** The list of bullets. */
  private List<Bullet> bullets = new LinkedList<Bullet>();
  /** The time this info was created. */
  private final long receiveTime;
  /** The time used for the last ping (or 0). */
  private final long pingTime;

  /**
   * Interprete 8 bits as unsigned byte.
   * @param b incomoing byte
   * @return unsigned interpretation
   */
  private static int byteToUnsigned(byte b)
  {
    return b < 0 ? b + 256 : b;
  }

  /**
   *  Constructor.
   *
   *  Extracts frame info from datagram.
   *  @param bytes      datagram bytes
   *  @param pingTimes  time stamps when pings were sent
   *  @throws IOException on i/o errors
   */
  FrameInfo(byte[] bytes, long[] pingTimes) throws IOException
  {
    if (bytes.length != MAME_DATAGRAM_SIZE) {
      throw new IOException("Incorrect datagram with size "+bytes.length);
    }
    if ((bytes[1] & MASK_OPCODE) != OPCODE_JMPL) {
      throw new IOException(String.format("Incorrect vector buffer start: %02x%02x", bytes[0], bytes[1]));
    }
    receiveTime = System.currentTimeMillis();
    id = bytes[ID_BYTE];
    ping = bytes[PING_BYTE];
    if (ping != Communication.NO_PING) {
      pingTime = receiveTime - pingTimes[ping];
    }
    else {
      pingTime = 0;
    }
    int vx = 0;
    int vy = 0;
    int vs = 0;
    int v1x = 0;
    int v1y = 0;
    int dx = 0;
    int dy = 0;
    boolean possibleShip = false;
    int p = 2;     // skip first two
    while (p < VECTORRAM_SIZE) {
      int highbyte = byteToUnsigned(bytes[p+1]);
      int opcode = highbyte & MASK_OPCODE;
      switch (opcode) {
      case OPCODE_LABS:
        vy = (highbyte & 0x03) << 8 | byteToUnsigned(bytes[p]);
        p += 2;
        highbyte = byteToUnsigned(bytes[p+1]);
        vx = (highbyte & 0x03) << 8 | byteToUnsigned(bytes[p]);
        vs = highbyte >> 4;
        p += 2;
        break;

      case OPCODE_HALT:
        // p += 2;
        return;

      case OPCODE_JSRL:
        switch ((highbyte & 0x0F) << 8  | byteToUnsigned(bytes[p])) {
        case SUB_ASTEROID_TYPE1:
          asteroids.add(new Asteroid(vx, vy, vs, 0));
          break;
        case SUB_ASTEROID_TYPE2:
          asteroids.add(new Asteroid(vx, vy, vs, 1));
          break;
        case SUB_ASTEROID_TYPE3:
          asteroids.add(new Asteroid(vx, vy, vs, 2));
          break;
        case SUB_ASTEROID_TYPE4:
          asteroids.add(new Asteroid(vx, vy, vs, 3));
          break;
        case SUB_UFO:
          ufo = new Ufo(vx, vy, vs);
          break;
        }
        p += 2;
        break;

      case OPCODE_RTSL:
        // p += 2;
        return;

      case OPCODE_JMPL:
        //p += 2;
        return;

      case OPCODE_SVEC:
        p += 2;
        break;

      default:
        if (spaceShip == null) {
          dy = (highbyte & 0x03) << 8 | byteToUnsigned(bytes[p]);
          if ((highbyte & 0x04) != 0) {
            dy = -dy;
          }
          p += 2;
          highbyte = byteToUnsigned(bytes[p+1]);
          dx = (highbyte & 0x03) << 8 | byteToUnsigned(bytes[p]);
          if ((highbyte & 0x04) != 0) {
            dx = -dx;
          }
          int vz = highbyte >> 4;
          if (dx == 0  &&  dy == 0) {
            if (vz == 15) {
              bullets.add(new Bullet(vx, vy));
            }
          }
          if (dx != 0  &&  dy != 0) {
            if (opcode == OPCODE_SHIP  &&  vz == 12) {
              if (possibleShip) {
                if (spaceShip == null) {
                  spaceShip = new SpaceShip(vx, vy, v1x - dx, v1y - dy);
                }
                possibleShip = false;
              }
              else {
                v1x = dx;
                v1y = dy;
                possibleShip = true;
              }
            }
          }
          else if (possibleShip) {
            possibleShip = false;
          }
          p += 2;
        }
        else {
          p += 4;
        }
        break;
      }
    }
  }

  /**
   *  Draw all game objects.
   *  @param g graphics context
   */
  public void draw(Graphics2D g)
  {
    for (Asteroid asteroid: asteroids) {
      asteroid.draw(g);
    }
    if (ufo != null) {
      ufo.draw(g);
    }
    if (spaceShip != null) {
      spaceShip.draw(g);
    }
    for (Bullet bullet: bullets) {
      bullet.draw(g);
    }
  }

  /**
   *  Get the frame id.
   *
   *  The frame id is a counter which is incremented by mame each time a frame is sent.
   *  @return frame id
   */
  public byte getId()
  {
    return id;
  }

  /**
   *  Get the ping associated with this frame.
   *  @return a number between <code>1</code> and <code>255</code> if there is a ping associated with this frame,
   *          otherwise <code>0</code>
   */
  public int getPing()
  {
    return ping;
  }

  /**
   *  Get the ping time associated with this frame.
   *  @return ping time or <code>0</code> if there is no ping associated with this frame
   */
  public long getPingTime()
  {
    return pingTime;
  }

  /**
   *  Get the time when this frame was received.
   *  @return receive time (compare System.currentTimeMillis())
   */
  public long getReceiveTime()
  {
    return receiveTime;
  }

  /**
   *  Get the ufo.
   *  @return ufo or <code>null</code> if no ufo is present
   */
  public Ufo getUfo()
  {
    return ufo;
  }

  /**
   *  Get the asteroids.
   *
   *  @return the asteroids
   */
  public Collection<Asteroid> getAsteroids()
  {
    return Collections.unmodifiableCollection(asteroids);
  }

  /**
   *  Get the space ship.
   *  @return space ship or <code>null</code> if there is no space ship present
   */
  public SpaceShip getSpaceShip()
  {
    return spaceShip;
  }

  /**
   *  Get the bullets.
   *
   *  @return bullets
   */
  public Collection<Bullet> getBullets()
  {
    return Collections.unmodifiableCollection(bullets);
  }
}