// ============================================================================
// File:               $File$
//
// Project:            
//
// Purpose:            
//
// Author:             Rammi
//
// Copyright Notice:   (c) 2008  Rammi (rammi@caff.de)
//                     This code is in the public domain.
//                     Use at own risk.
//                     No guarantees given.
//
// Latest change:      $Date$
//
// History:	       $Log$
//=============================================================================
package de.caff.asteroid;

import java.util.*;

/**
 *  Helper class which tries to add velocities to game objects.
 *  This is done in a too simple-minded way by just comparing the last two screens.
 *
 *  This class is part of a solution for a
 *  <a href="http://www.heise.de/ct/creativ/08/02/details/">competition by the German computer magazine c't</a>.
 */
public class SimpleVelocityPreparer
        implements FramePreparer
{
  /**
   *  Helper class to presort asteroids to allow easier compare between frames.
   */
  private static class AsteroidSelector
  {
    /** Maximum squared distance we assume an asteroid may move between frames. */
    private static final int MAX_SQUARED_DIST = 64*64;

    /**
     *  Helper class: key for mapping asteroids.
     */
    private static class Key
    {
      /** The asteroid's type. */
      private final int type;
      /** The asteroids size. */
      private final int size;

      /**
       * Constrictor.
       * @param ast  asteroid
       */
      private Key(Asteroid ast)
      {
        this.type = ast.getType();
        this.size = ast.getSize();
      }

      public boolean equals(Object o)
      {
        if (this == o) {
          return true;
        }
        if (o == null || getClass() != o.getClass()) {
          return false;
        }

        Key key = (Key)o;

        if (size != key.size) {
          return false;
        }
        if (type != key.type) {
          return false;
        }

        return true;
      }

      public int hashCode()
      {
        int result;
        result = type;
        result = 31 * result + size;
        return result;
      }
    }
    /** Presorted mapping of asteroids. */
    private Map<Key, Collection<Asteroid>> sorted = new HashMap<Key, Collection<Asteroid>>();

    /**
     *  Constructor.
     *  @param asteroids asteroids to be presorted
     */
    private AsteroidSelector(Collection<Asteroid> asteroids)
    {
      for (Asteroid ast: asteroids) {
        Key key = new Key(ast);
        Collection<Asteroid> list = sorted.get(key);
        if (list == null) {
          list = new LinkedList<Asteroid>();
          sorted.put(key, list);
        }
        list.add(ast);
      }
    }

    /**
     *  Get the best matching asteroid assuming the keys collected here are from the previous
     *  frame while the given asteroid is from the current frame.
     *  @param asteroid current frame astreroid
     *  @return best match or <code>null</code>
     */
    public Asteroid getBestMatch(Asteroid asteroid)
    {
      Key key = new Key(asteroid);
      Collection<Asteroid> list = sorted.get(key);
      SortedMap<Integer, Asteroid> result = new TreeMap<Integer, Asteroid>();
      if (list != null) {
        int futureX = asteroid.getX() + asteroid.getVelocityX();
        int futureY = asteroid.getY() + asteroid.getVelocityY();
        for (Asteroid a: list) {
          int dist2 = a.getSquaredDistance(futureX, futureY);
          if (dist2 < MAX_SQUARED_DIST) {
            result.put(dist2, a);
          }
        }
      }
      return result.isEmpty() ? null : result.values().iterator().next();
    }
  }

  /**
   * Prepare the frame(s).
   *
   * @param frameInfos the collected frame infos
   */
  public void prepareFrames(LinkedList<FrameInfo> frameInfos)
  {
    if (frameInfos.size() >= 2) {
      FrameInfo lastFrame = frameInfos.getLast();
      FrameInfo prevFrame = frameInfos.get(frameInfos.size() - 2);
      Ufo ufo = lastFrame.getUfo();
      if (ufo != null) {
        ufo.setVelocityFromDelta(prevFrame.getUfo());
      }
      SpaceShip ship = lastFrame.getSpaceShip();
      if (ship != null) {
        ship.setVelocityFromDelta(prevFrame.getSpaceShip());
      }
      AsteroidSelector selector = new AsteroidSelector(lastFrame.getAsteroids());

      for (Asteroid ast: prevFrame.getAsteroids()) {
        Asteroid candidate = selector.getBestMatch(ast);
        if (candidate != null) {
          candidate.setVelocityFromDelta(ast);
        }
      }
    }
  }
}
