// ============================================================================
// File:               $File$
//
// Project:            Asteroids game solver.
//
// Purpose:            
//
// Author:             Rammi
//
// Copyright Notice:   (c) 2008  Rammi (rammi@caff.de)
//                     This code is in the public domain.
//                     Use at own risk.
//                     No guarantees given.
//
// Latest change:      $Date$
//
// History:	       $Log$
//=============================================================================
package de.caff.asteroid;

import java.awt.*;
import java.awt.geom.AffineTransform;
import java.awt.geom.GeneralPath;

/**
 * Asteroid object in automated player of Asteroids game.
 *
 * This class is part of a solution for a
 * <a href="http://www.heise.de/ct/creativ/08/02/details/">competition by the German computer magazine c't</a>.
 */
public class Asteroid
        extends SizedGameObject
{
  /** Scaling for shapes when drawing big asteroid. */
  private int NORMAL_SCALE = 8;
  /** Scaling of this asteroid. */
  private int scale;
  /** Type of this asteroid. There are 4 different types numbered from 0 to 3. */
  private int type;

  private static final GeneralPath PATH0 = new GeneralPath();
  private static final GeneralPath PATH1 = new GeneralPath();
  private static final GeneralPath PATH2 = new GeneralPath();
  private static final GeneralPath PATH3 = new GeneralPath();
  static {
    PATH0.moveTo(1, 0);
    PATH0.lineTo(4, 1);
    PATH0.lineTo(4, 2);
    PATH0.lineTo(1, 4);
    PATH0.lineTo(-2, 4);
    PATH0.lineTo(-1, 3);
    PATH0.lineTo(-4, 3);
    PATH0.lineTo(-4, -1);
    PATH0.lineTo(-2, -4);
    PATH0.lineTo(1, -3);
    PATH0.lineTo(2, -4);
    PATH0.lineTo(4, -2);
    PATH0.closePath();

    PATH1.moveTo(3, 0);
    PATH1.lineTo(4, 2);
    PATH1.lineTo(2, 4);
    PATH1.lineTo(0, 2);
    PATH1.lineTo(-2, 4);
    PATH1.lineTo(-4, 2);
    PATH1.lineTo(-4, -2);
    PATH1.lineTo(-2, -4);
    PATH1.lineTo(1, -4);
    PATH1.lineTo(4, -2);
    PATH1.closePath();

    PATH2.moveTo(4, 1);
    PATH2.lineTo(2, 4);
    PATH2.lineTo(-1, 4);
    PATH2.lineTo(-4, 1);
    PATH2.lineTo(-2, 0);
    PATH2.lineTo(-4, -1);
    PATH2.lineTo(-2, -4);
    PATH2.lineTo(0, -1);
    PATH2.lineTo(0, -4);
    PATH2.lineTo(2, -4);
    PATH2.lineTo(4, -1);
    PATH2.closePath();

    PATH3.moveTo(2, 1);
    PATH3.lineTo(4, 2);
    PATH3.lineTo(2, 4);
    PATH3.lineTo(0, 3);
    PATH3.lineTo(-2, 4);
    PATH3.lineTo(-4, 2);
    PATH3.lineTo(-3, 0);
    PATH3.lineTo(-4, -2);
    PATH3.lineTo(-2, -4);
    PATH3.lineTo(-1, -3);
    PATH3.lineTo(2, -4);
    PATH3.lineTo(4, -1);
    PATH3.closePath();
  }
  /** The asteroid shapes sorted by type. */
  private static final Shape[] SHAPES = new Shape[] {
          PATH1,
          PATH3,
          PATH2,
          PATH0,
  };

  /**
   * Constructor.
   * @param x position (x coordinate)
   * @param y position (y coordinate)
   * @param size size (as given by game, so 0 is big and 14 is small)
   * @param type type of asteroid (0, 1, 2, or 3)
   */
  public Asteroid(int x, int y, int size, int type)
  {
    super(x, y);
    switch (size) {
    case 0:
      this.scale = NORMAL_SCALE;
      break;

    case 15:
      this.scale = NORMAL_SCALE/2;
      break;

    case 14:
      this.scale = NORMAL_SCALE/4;
      break;
    }
    assert(type >= 0  &&  type < SHAPES.length);
    this.type = type;
  }

  /**
   *  Get the size of the object.
   *
   *  The size returned by this method is half the length of a square which contains the object,
   *  so the object's bounding box is between (x - size, y - size) and (x + size, y + size).
   *  @return object size
   */
  public int getSize()
  {
    return 4*scale;
  }

  public int getType()
  {
    return type;
  }

  /**
   *  Draw the object.
   *  @param g graphics context
   */
  @Override
  public void draw(Graphics2D g)
  {
    g.setColor(Color.gray);
    AffineTransform trafo = new AffineTransform(scale, 0, 0, scale, x, y);
    g.draw(trafo.createTransformedShape(SHAPES[type]));

    drawVelocityVector(g, Color.red);
  }

  /**
   * Returns a string representation of the object.
   *
   * @return a string representation of the object.
   */
  @Override
  public String toString()
  {
    return String.format("Asteroid@(%d,%d)*%d?%d", x, y, 4*scale, type);
  }


  private static int minVX = 0;
  private static int maxVX = 0;
  private static int minVY = 0;
  private static int maxVY = 0;
  private static int maxVSquared   = 0;

  /**
   * Set the velocity.
   * <p/>
   * The velocity is the step between frames.
   *
   * @param x x coordinate of velocity
   * @param y y coordinate of velocity
   */
  @Override
  public void setVelocity(int x, int y)
  {
    super.setVelocity(x, y);
    if (WATCH_VELOCITIES) {
      boolean print = false;
      if (x < minVX) {
        minVX = x;
        print = true;
      }
      else if (x > maxVX) {
        maxVX = x;
        print = true;
      }
      if (y < minVY) {
        minVY = y;
        print = true;
      }
      else if (y > maxVY) {
        maxVY = y;
        print = true;
      }
      int len2 = x*x + y*y;
      if (len2 > maxVSquared) {
        maxVSquared = len2;
        print = true;
      }
      if (print) {
        System.out.println(String.format("AST: minV=(%d,%d), maxV=(%d,%d), maxV^2=%d",
                                         minVX, minVY, maxVX, maxVY, maxVSquared));
      }
    }
  }
}

