// ============================================================================
// File:               $File$
//
// Project:            
//
// Purpose:            
//
// Author:             Rammi
//
// Copyright Notice:   (c) 2008  Rammi (rammi@caff.de)
//                     This code is in the public domain.
//                     Use at own risk.
//                     No guarantees given.
//
// Latest change:      $Date$
//
// History:	       $Log$
//=============================================================================
package de.caff.asteroid;

import javax.swing.*;
import java.awt.*;
import java.awt.geom.AffineTransform;

/**
 *  Component to display a frame as received from MAME.
 *
 *  Because this an example of a frame listener note the usage of synchronized access to
 *  the frameInfo object passed between threads and the usage of repaint() to inform the AWT
 *  thread to start drawing.
 *
 *  This class is part of a solution for a
 *  <a href="http://www.heise.de/ct/creativ/08/02/details/">competition by the German computer magazine c't</a>
 */
public class FrameDisplay
  extends JComponent
  implements FrameListener,
             GameData
{
  /** The currently displayed frame info. */
  private FrameInfo frameInfo;
  /** The transformation from frame space into component space. */
  private AffineTransform trafo;

  /**
   *  Constructor.
   *
   *  The component is always constructed with a aspect ratio of 4:3.
   *  @param width width to use for this component
   */
  public FrameDisplay(int width)
  {
    Dimension size = new Dimension(width, 3*width/4);
    setMaximumSize(size);
    setMinimumSize(size);
    setPreferredSize(size);
    setOpaque(true);
    setDoubleBuffered(true);

    double scaling = width/(double)EXTENT;
    // NOTE: take care of y pointing upwards in Asteroids, but downwards on screen
    trafo = AffineTransform.getTranslateInstance(0, MIN_Y-EXTENT);
    trafo.preConcatenate(AffineTransform.getScaleInstance(scaling, -scaling));
  }

  /**
   * Invoked by Swing to draw components.
   * Applications should not invoke <code>paint</code> directly,
   * but should instead use the <code>repaint</code> method to
   * schedule the component for redrawing.
   * <p/>
   * This method actually delegates the work of painting to three
   * protected methods: <code>paintComponent</code>,
   * <code>paintBorder</code>,
   * and <code>paintChildren</code>.  They're called in the order
   * listed to ensure that children appear on top of component itself.
   * Generally speaking, the component and its children should not
   * paint in the insets area allocated to the border. Subclasses can
   * just override this method, as always.  A subclass that just
   * wants to specialize the UI (look and feel) delegate's
   * <code>paint</code> method should just override
   * <code>paintComponent</code>.
   *
   * @param g the <code>Graphics</code> context in which to paint
   * @see #paintComponent
   * @see #paintBorder
   * @see #paintChildren
   * @see #getComponentGraphics
   * @see #repaint
   */
  @Override
  public void paint(Graphics g)
  {
    g.setColor(Color.black);
    g.fillRect(0, 0, getWidth(), getHeight());
    FrameInfo info;
    synchronized (this) {
      info = frameInfo;
    }
    if (info != null) {
      Graphics2D g2 = (Graphics2D)g.create();
      g2.transform(trafo);
      info.draw(g2);
      Point pos = new Point();
      if (!info.isGameRunning()) {
        g.setColor(new Color(0xFF, 0xFF, 0x00, 0x80));
        for (Text txt: info.getTexts()) {
          trafo.transform(txt.getLocation(), pos);
          g.drawString(txt.getText(), pos.x, pos.y);
        }
      }
      // High score
      g.setColor(new Color(0xFF, 0xFF, 0xFF, 0x80));
      trafo.transform(FrameInfo.SCORE_LOCATION_GAME, pos);
      g.drawString(String.format("%7d", info.getScore()),
                   pos.x, pos.y);
    }
  }

  /**
   *  Called each time a frame is received.
   *
   *  <b>ATTENTION:</b> this is called from the communication thread!
   *  Implementing classes must be aware of this and take care by synchronization or similar!
   *  @param frame the received frame
   */
  public void frameReceived(FrameInfo frame)
  {
    synchronized (this) {
      frameInfo = frame;
    }
    repaint();
  }

}
