// ============================================================================
// File:               $File$
//
// Project:            
//
// Purpose:            
//
// Author:             Rammi
//                     This code is in the public domain.
//                     Use at own risk.
//                     No guarantees given.
//
// Copyright Notice:   (c) 2008  Rammi (rammi@caff.de)
//
// Latest change:      $Date$
//
// History:	       $Log$
//=============================================================================
package de.caff.asteroid;

import de.caff.asteroid.server.DatagramListener;
import de.caff.asteroid.server.DatagramSender;

import java.io.IOException;
import java.net.*;
import java.util.*;

/**
 * UDP communication with MAME.
 *
 * This class is part of a solution for a
 * <a href="http://www.heise.de/ct/creativ/08/02/details/">competition by the German computer magazine c't</a>.
 */
public class Communication
        implements Runnable,
                   GameData,
                   DatagramSender
{

  /** Maximum capacity of the pendingFrames fifo. */
  public static final int MAX_FRAMES_KEPT = 256;

  /** Socket info. */
  private DatagramSocket socket;
  /** FIFO of frames, sorted in the order they are received. The maximum capacity is kept to {@link #MAX_FRAMES_KEPT}. */
  private LinkedList<FrameInfo> pendingFrames = new LinkedList<FrameInfo>();
  /** Reused byte array for sending the keys. */
  private byte[] keyFrame = new byte[KEY_PACKET_SIZE];
  {
    System.arraycopy(KEY_PACKET_INTRO, 0, keyFrame, 0, KEY_PACKET_INTRO.length);
  }
  /** Reused datagram for sending the keys. */
  private DatagramPacket keyDatagram = new DatagramPacket(keyFrame, keyFrame.length);
  /** Reused datagram for frame data receive. */
  private DatagramPacket mameDatagram = new DatagramPacket(new byte[FrameInfo.MAME_DATAGRAM_SIZE], FrameInfo.MAME_DATAGRAM_SIZE);
  /** Registered frame listeners. */
  private List<FrameListener> frameListeners = new LinkedList<FrameListener>();
  /** Registered datagram listeners. */
  private List<DatagramListener> datagramListeners = new LinkedList<DatagramListener>();
  /** The ping of the next key datagram. */
  private int ping = 1;
  /** The ping of the latest key datagram. */
  private int latestPing = 0;
  /** Times when we sent key datagrams, sorted by the associated ping. */
  private long[] pingTimes = new long[256];
  /** Address of MAME. */
  private InetSocketAddress mameAddr;
  /** Frame preparer. */
  private FramePreparer framePreparer;
  /** The buttons. */
  private Buttons buttons = new Buttons();
  /** Brigde mode? */
  private boolean bridgeMode;

  /**
   * Constructor.
   * @param hostname hostname to use
   * @param bridgeMode run in bridge mode (then all key-related methods are useless,
   *                   because no keys datagrams are send to MAME)
   * @throws SocketException on connection problems
   */
  public Communication(String hostname, boolean bridgeMode) throws IOException
  {
    this.bridgeMode = bridgeMode;
    int port = MAME_PORT;
    String[] parts = hostname.split(":");
    if (parts.length == 2) {
      try {
        port = Integer.parseInt(parts[1]);
        hostname = parts[0];
      } catch (NumberFormatException e) {
        e.printStackTrace();
      }
    }
    socket = new DatagramSocket();
    mameAddr = new InetSocketAddress(hostname, port);
    socket.connect(mameAddr);
  }

  /**
   * Start thread.
   * @see Thread#run()
   */
  public void run()
  {
    try {
      while (true) {
        if (!bridgeMode) {
          sendKeys();
        }
        socket.receive(mameDatagram);
        long receiveTime = System.currentTimeMillis();
        informDatagramListeners(mameDatagram);
        FrameInfo fi = new FrameInfo(mameDatagram.getData(), pingTimes, receiveTime);
        synchronized (pendingFrames) {
          if (!pendingFrames.isEmpty()) {
            if (pendingFrames.size() >= MAX_FRAMES_KEPT) {
              pendingFrames.removeFirst();
            }
            if (true) {
              byte lastId = pendingFrames.getLast().getId();
              byte id     = fi.getId();
              if ((byte)(id - lastId) != 1) {
                System.err.println("Frame dropped? last="+lastId+", current="+id);
              }
            }
            if (framePreparer != null) {
              framePreparer.prepareFrames(pendingFrames);
            }
          }
          pendingFrames.add(fi);
          if (framePreparer != null) {
            framePreparer.prepareFrames(pendingFrames);
          }
        }
        informFrameListeners(fi);
      }
    } catch (IOException e) {
      e.printStackTrace();
    }
  }

  /**
   *  Send the pressed keys.
   *
   *  @throws IOException if sending failed
   */
  private void sendKeys() throws IOException
  {
    synchronized (keyDatagram) {
      keyFrame[KEY_MASK_INDEX] = buttons.extractKeys();
      keyFrame[KEY_PING_INDEX] = (byte)ping;
      keyDatagram.setData(keyFrame);
      pingTimes[ping] = System.currentTimeMillis();
      socket.send(keyDatagram);
      informDatagramListeners(keyDatagram);
      latestPing = ping;
      if (++ping >= 256) {
        ping = 1;
      }
    }
  }

  /**
   *  Set a button.
   *  @param button the button (one of
   *                {@link #BUTTON_FIRE},
   *                {@link #BUTTON_HYPERSPACE},
   *                {@link #BUTTON_LEFT},
   *                {@link #BUTTON_RIGHT},
   *                {@link #BUTTON_THRUST}, and possibly
   *                {@link #BUTTON_START} (requires patched MAME version)
   *  @param down button down?
   */
  public void setButton(int button, boolean down)
  {
    buttons.setKey(button, down);
  }

  /**
   *  Set a button down.
   *  @param button the button (one of
   *                {@link #BUTTON_FIRE},
   *                {@link #BUTTON_HYPERSPACE},
   *                {@link #BUTTON_LEFT},
   *                {@link #BUTTON_RIGHT},
   *                {@link #BUTTON_THRUST}, and possibly
   *                {@link #BUTTON_START} (requires patched MAME version))
   */
  public void pushButton(int button)
  {
    setButton(button, true);
  }

  /**
   *  Get the currently pressed buttons.
   *  @return button mask (a combination of
   *                {@link #BUTTON_FIRE},
   *                {@link #BUTTON_HYPERSPACE},
   *                {@link #BUTTON_LEFT},
   *                {@link #BUTTON_RIGHT},
   *                {@link #BUTTON_THRUST}, and possibly
   *                {@link #BUTTON_START} (requires patched MAME version))
   */
  public byte getButtons()
  {
    return buttons.getKeys();
  }

  /**
   *  Add a frame listener which is called on every frame received.
   *  @param listener listener to add
   */
  public void addFrameListener(FrameListener listener)
  {
    synchronized (frameListeners) {
      frameListeners.add(listener);
    }
  }

  /**
   *  Remove a frame listener.
   *  @param listener listener to remove
   *  @return <code>true</code> if the listener was removed<br>
   *          <code>false</code> otherwise
   */
  public boolean removeFrameListener(FrameListener listener)
  {
    synchronized (frameListeners) {
      return frameListeners.remove(listener);
    }
  }

  /**
   *  Add a datagram listener which is called on every datagram received.
   *  @param listener listener to add
   */
  public void addDatagramListener(DatagramListener listener)
  {
    synchronized (datagramListeners) {
      datagramListeners.add(listener);
    }
  }

  /**
   *  Remove a datagram listener.
   *  @param listener listener to remove
   *  @return <code>true</code> if the listener was removed<br>
   *          <code>false</code> otherwise
   */
  public boolean removeDatagramListener(DatagramListener listener)
  {
    synchronized (datagramListeners) {
      return datagramListeners.remove(listener);
    }
  }

  /**
   *  Inform all registered frame listeners.
   *  @param frameInfo frame info received
   */
  private void informFrameListeners(FrameInfo frameInfo)
  {
    Collection<FrameListener> tmp;
    synchronized (frameListeners) {
      if (frameListeners.isEmpty()) {
        return;
      }
      tmp = new ArrayList<FrameListener>(frameListeners);
    }
    for (FrameListener listener: tmp) {
      listener.frameReceived(frameInfo);
    }
  }

  /**
   *  Inform all registered datagram listeners.
   *  @param datagram datagram received
   */
  private void informDatagramListeners(DatagramPacket datagram)
  {
    Collection<DatagramListener> tmp;
    synchronized (datagramListeners) {
      if (datagramListeners.isEmpty()) {
        return;
      }
      tmp = new ArrayList<DatagramListener>(datagramListeners);
    }
    for (DatagramListener listener: tmp) {
      listener.datagramReceived(datagram, this);
    }
  }

  /**
   *  Send a datagram packet.
   *  @param packet packet to send
   *  @throws IOException on send errors
   */
  public void sendDatagram(DatagramPacket packet) throws IOException
  {
    packet.setSocketAddress(mameAddr);
    socket.send(packet);
    informDatagramListeners(packet);
  }

  /**
   * Get the collected frames.
   *
   * This class collects only the last 256 frames, older frames are discarded.
   * @return the last frames in the sequence in which they were received (but not necessarily send)
   */
  public Collection<FrameInfo> getFrames()
  {
    synchronized (pendingFrames) {
      return new ArrayList<FrameInfo>(pendingFrames);
    }
  }

  /**
   *  Get all frames with a newer timestamp.
   *  @param timestamp timestamp
   *  @return collection of frames with a receive time newer than the given timestamp
   */
  public Collection<FrameInfo> getFramesAfter(long timestamp)
  {
    // NOTE:
    // This implementation assumes that usually the timestamp selects only
    // the last few frames. It assumes that the pending frames are in the sequence
    // in which they were received, so if you ever change that (e.g. because you
    // are reordering by sequence numbers) you have to reimplement this method!
    ArrayList<FrameInfo> tmpFrames;
    synchronized (pendingFrames) {
      if (pendingFrames.isEmpty()  ||  pendingFrames.getFirst().getReceiveTime() > timestamp) {
        return getFrames();
      }
      tmpFrames = new ArrayList<FrameInfo>(pendingFrames);
    }
    LinkedList<FrameInfo> result = new LinkedList<FrameInfo>();
    for (int f = tmpFrames.size() - 1;  f >= 0;  --f) {
      FrameInfo info = tmpFrames.remove(f);
      if (info.getReceiveTime() <= timestamp) {
        break;
      }
      result.add(0, info);
    }
    return result;
  }

  /**
   *  Get the frame preparer.
   *  @return frame preparer or <code>null</code> if there is no frame preparer
   */
  public FramePreparer getFramePreparer()
  {
    return framePreparer;
  }

  /**
   *  Set the frame preparer.
   *
   *  The frame preparer is called on each frame as it is received before the frame
   *  listeners are called. It is most useful to create connections between frames, e.g.
   *  to calculate velocities are make any other connections between objects in different
   *  frames. See {@link de.caff.asteroid.SimpleVelocityPreparer} for example.
   *  @param framePreparer frame peparer to set or <code>null</code> to unset frame preparing
   */
  public void setFramePreparer(FramePreparer framePreparer)
  {
    this.framePreparer = framePreparer;
  }

  /**
   *  Get the ping sent with the latest key datagram.
   *  @return latest ping
   */
  public int getLatestPing()
  {
    return latestPing;
  }
}