// ============================================================================
// File:               $File$
//
// Project:            
//
// Purpose:            
//
// Authors:            Harald Boegeholz (original C++ code)
//                     Rammi (port to Java and more)
//
// Copyright Notice:   (c) 2008  Harald Boegehoz/c't
//                     (c) 2008  Rammi (rammi@caff.de)
//                     This code is in the public domain.
//                     Use at own risk.
//                     No guarantees given.
//
// Latest change:      $Date$
//
// History:	       $Log$
//=============================================================================
package de.caff.asteroid;

import de.caff.util.Tools;

import java.awt.*;
import java.io.IOException;
import java.util.*;
import java.util.List;

/**
 *  The info describing one game frame constructed from a received datagram.
 *
 *  This class is part of a solution for a
 *  <a href="http://www.heise.de/ct/creativ/08/02/details/">competition by the German computer magazine c't</a>.
 */
public class FrameInfo
        implements GameData,
                   PropertyProvider
{
  /** Mapping of ROM adresses to characters (used by {@link #canonize(String)}, there placed before string constants). */
  private static final Map<Integer, Character> CHAR_MAP = new HashMap<Integer, Character>();
  static {
    CHAR_MAP.put(0xA78, 'A');
    CHAR_MAP.put(0xA80, 'B');
    CHAR_MAP.put(0xA8D, 'C');
    CHAR_MAP.put(0xA93, 'D');
    CHAR_MAP.put(0xA9B, 'E');
    CHAR_MAP.put(0xAA3, 'F');
    CHAR_MAP.put(0xAAA, 'G');
    CHAR_MAP.put(0xAB3, 'H');
    CHAR_MAP.put(0xABA, 'I');
    CHAR_MAP.put(0xAC1, 'J');
    CHAR_MAP.put(0xAC7, 'K');
    CHAR_MAP.put(0xACD, 'L');
    CHAR_MAP.put(0xAD2, 'M');
    CHAR_MAP.put(0xAD8, 'N');
    CHAR_MAP.put(0xADD, '0'); // zero instead if oh!
    CHAR_MAP.put(0xAE3, 'P');
    CHAR_MAP.put(0xAEA, 'Q');
    CHAR_MAP.put(0xAF3, 'R');
    CHAR_MAP.put(0xAFB, 'S');
    CHAR_MAP.put(0xB02, 'T');
    CHAR_MAP.put(0xB08, 'U');
    CHAR_MAP.put(0xB0E, 'V');
    CHAR_MAP.put(0xB13, 'W');
    CHAR_MAP.put(0xB1A, 'X');
    CHAR_MAP.put(0xB1F, 'Y');
    CHAR_MAP.put(0xB26, 'Z');
    CHAR_MAP.put(0xB2C, ' ');
    CHAR_MAP.put(0xB2E, '1');
    CHAR_MAP.put(0xB32, '2');
    CHAR_MAP.put(0xB3A, '3');
    CHAR_MAP.put(0xB41, '4');
    CHAR_MAP.put(0xB48, '5');
    CHAR_MAP.put(0xB4F, '6');
    CHAR_MAP.put(0xB56, '7');
    CHAR_MAP.put(0xB5B, '8');
    CHAR_MAP.put(0xB63, '9');
  }

  /** Game end string, may differ with other (non-German) ROMs.
   *  If changing use only uppercase letters and 0 (zero) instead of O (oh).
   */
  public static final String GAME_END_STRING = canonize("SPIELENDE");

  /** Game start string, may differ with other (non-German) ROMs.
   *  If changing use only uppercase letters and 0 (zero) instead of O (oh).
   */
  public static final String GAME_START_STRING = canonize("STARTKN0EPFE DRUECKEN");

  /** Game starting string, may differ with other (non-German) ROMs.
   *  If changing use only uppercase letters and 0 (zero) instead of O (oh).
   */
  public static final String PLAYER1_STRING = canonize("SPIELER 1");

  /** Return value of {@link #extractNumber(String)} indicating no valid number . */
  private static final int NO_NUMBER = -1;

  /** Position of id byte in incoming datagram. */
  private static final int ID_BYTE = VECTORRAM_SIZE;
  /** Position of ping byte in incoming datagram. */
  private static final int PING_BYTE = ID_BYTE+1;

  private static final int MASK_OPCODE = 0xF0;
  private static final int OPCODE_LABS = 0xA0;
  private static final int OPCODE_HALT = 0xB0;
  private static final int OPCODE_JSRL = 0xC0;
  private static final int OPCODE_RTSL = 0xD0;
  private static final int OPCODE_JMPL = 0xE0;
  private static final int OPCODE_SVEC = 0xF0;
  private static final int OPCODE_SHIP = 0x60;

  private static final int SUB_ASTEROID_TYPE1 = 0x8F3;
  private static final int SUB_ASTEROID_TYPE2 = 0x8FF;
  private static final int SUB_ASTEROID_TYPE3 = 0x90D;
  private static final int SUB_ASTEROID_TYPE4 = 0x91A;
  private static final int SUB_UFO            = 0x929;

  private static final int SUB_SHIP = 0xA6D;
  private static final int SUB_EXPLOSION_XXL = 0x880;
  private static final int SUB_EXPLOSION_XL  = 0x896;
  private static final int SUB_EXPLOSION_L   = 0x8B5;
  private static final int SUB_EXPLOSION_S   = 0x8D0;

  private static int FRAME_COUNT = 0;

  /** The counter (index) of this frame. */
  private int counter = FRAME_COUNT++;
  /** Frame ID as sent by MAME. */
  private final byte id;
  /** Ping ID as sent by MAME. */
  private final byte ping;
  /** The ufo (if any). */
  private Ufo ufo;
  /** The list of asteroids. */
  private List<Asteroid> asteroids = new LinkedList<Asteroid>();
  /** The spaceship (if any). */
  private SpaceShip spaceShip;
  /** The list of bullets. */
  private List<Bullet> bullets = new LinkedList<Bullet>();
  /** The list of explosions. */
  private List<Explosion> explosions = new LinkedList<Explosion>();
  /** The time this info was created. */
  private final long receiveTime;
  /** The time used for the last ping (or 0). */
  private final long pingTime;
  /** The score (upper right corner). */
  private int score = NO_NUMBER;
  /** The high score (central top). */
  private int highscore = NO_NUMBER;
  /** Some other number in right top. */
  private int sonirt = NO_NUMBER;
  /** Game start visible. */
  private boolean gameStartDisplayed;
  /** Player1 visible. */
  private boolean player1Displayed;
  /** Game end visible. */
  private boolean gameEndDisplayed;
  /** Number of ships (upper right corner). */
  private int nrShips;
  /** Text displayed on screen. */
  private List<Text> texts = new LinkedList<Text>();

  /**
   *  Constructor.
   *
   *  Extracts frame info from datagram.
   *  @param bytes      datagram bytes
   *  @param pingTimes  time stamps when pings were sent
   *  @param receiveTime time when this frame was received
   *  @throws IOException on i/o errors
   */
  public FrameInfo(byte[] bytes, long[] pingTimes, long receiveTime) throws IOException
  {
    if (bytes.length != MAME_DATAGRAM_SIZE) {
      throw new IOException("Incorrect datagram with size "+bytes.length);
    }
    if ((bytes[1] & MASK_OPCODE) != OPCODE_JMPL) {
      throw new IOException(String.format("Incorrect vector buffer start: %02x%02x", bytes[0], bytes[1]));
    }
    this.receiveTime = receiveTime;
    id = bytes[ID_BYTE];
    ping = bytes[PING_BYTE];
    if (ping != Communication.NO_PING) {
      pingTime = receiveTime - pingTimes[Tools.byteToUnsigned(ping)];
    }
    else {
      pingTime = 0;
    }
    int vx = 0;
    int vy = 0;
    int vs = 0;
    int v1x = 0;
    int v1y = 0;
    int dx = 0;
    int dy = 0;
    int tx = 0;
    int ty = 0;
    int ts = 0;
    int astIdx = 0;
    boolean possibleShip = false;
    StringBuilder stringCollect = new StringBuilder();
    int p = 2;     // skip first two
    while (p < VECTORRAM_SIZE) {
      boolean addedLetter = false;
      int highbyte = Tools.byteToUnsigned(bytes[p+1]);
      int opcode = highbyte & MASK_OPCODE;
      switch (opcode) {
      case OPCODE_LABS:
        vy = (highbyte & 0x03) << 8 | Tools.byteToUnsigned(bytes[p]);
        p += 2;
        highbyte = Tools.byteToUnsigned(bytes[p+1]);
        vx = (highbyte & 0x03) << 8 | Tools.byteToUnsigned(bytes[p]);
        vs = highbyte >> 4;
        p += 2;
        break;

      case OPCODE_HALT:
        // p += 2;
        return;

      case OPCODE_JSRL:
        int sub = (highbyte & 0x0F) << 8  | Tools.byteToUnsigned(bytes[p]);
        switch (sub) {
        case SUB_ASTEROID_TYPE1:
          asteroids.add(new Asteroid(astIdx++, vx, vy, vs, 0));
          break;
        case SUB_ASTEROID_TYPE2:
          asteroids.add(new Asteroid(astIdx++, vx, vy, vs, 1));
          break;
        case SUB_ASTEROID_TYPE3:
          asteroids.add(new Asteroid(astIdx++, vx, vy, vs, 2));
          break;
        case SUB_ASTEROID_TYPE4:
          asteroids.add(new Asteroid(astIdx++, vx, vy, vs, 3));
          break;
        case SUB_UFO:
          ufo = new Ufo(vx, vy, vs);
          break;
        case SUB_EXPLOSION_XXL:
          explosions.add(new Explosion(vx, vy, vs, Explosion.Type.XXL));
          break;
        case SUB_EXPLOSION_XL:
          explosions.add(new Explosion(vx, vy, vs, Explosion.Type.XL));
          break;
        case SUB_EXPLOSION_L:
          explosions.add(new Explosion(vx, vy, vs, Explosion.Type.L));
          break;
        case SUB_EXPLOSION_S:
          explosions.add(new Explosion(vx, vy, vs, Explosion.Type.S));
          break;
        case SUB_SHIP:
          nrShips++;
        default:
          // look for letters
          Character ch = CHAR_MAP.get(sub);
          if (ch != null) {
            stringCollect.append(ch);
            if (!addedLetter) {
              addedLetter = true;
              tx = vx;
              ty = vy;
              ts = vs;
            }
          }
        }
        p += 2;
        break;

      case OPCODE_RTSL:
        // p += 2;
        return;

      case OPCODE_JMPL:
        //p += 2;
        return;

      case OPCODE_SVEC:
        p += 2;
        break;

      default:
        if (spaceShip == null) {
          dy = (highbyte & 0x03) << 8 | Tools.byteToUnsigned(bytes[p]);
          if ((highbyte & 0x04) != 0) {
            dy = -dy;
          }
          p += 2;
          highbyte = Tools.byteToUnsigned(bytes[p+1]);
          dx = (highbyte & 0x03) << 8 | Tools.byteToUnsigned(bytes[p]);
          if ((highbyte & 0x04) != 0) {
            dx = -dx;
          }
          int vz = highbyte >> 4;
          if (dx == 0  &&  dy == 0) {
            if (vz == 15) {
              bullets.add(new Bullet(vx, vy));
            }
          }
          if (dx != 0  &&  dy != 0) {
            if (opcode == OPCODE_SHIP  &&  vz == 12) {
              if (possibleShip) {
                if (spaceShip == null) {
                  spaceShip = new SpaceShip(vx, vy, v1x - dx, v1y - dy);
                }
                possibleShip = false;
              }
              else {
                v1x = dx;
                v1y = dy;
                possibleShip = true;
              }
            }
          }
          else if (possibleShip) {
            possibleShip = false;
          }
          p += 2;
        }
        else {
          p += 4;
        }
        break;
      }
      if (!addedLetter  &&  stringCollect.length() > 0) {
        // collected string
        String str = stringCollect.toString();
        texts.add(new Text(str, tx, ty, ts));
        stringCollect.setLength(0);
        if ((tx == SCORE_LOCATION_GAME.x  &&  ty == SCORE_LOCATION_GAME.y)  ||
                (tx == SCORE_LOCATION_OTHER.x  &&  ty == SCORE_LOCATION_OTHER.y)) {
          score = extractNumber(str);
        }
        else if (tx == HIGHSCORE_LOCATION.x  &&  ty == HIGHSCORE_LOCATION.y) {
          highscore = extractNumber(str);
        }
        else if (tx == SONIRT_LOCATION.x  &&  ty == SONIRT_LOCATION.y) {
          sonirt = extractNumber(str);
        }
        else if (GAME_END_STRING.equals(str)) {
          gameEndDisplayed = true;
        }
        else if (GAME_START_STRING.equals(str)) {
          gameStartDisplayed = true;
        }
        else if (PLAYER1_STRING.equals(str)) {
          player1Displayed = true;
        }
      }
    }
  }

  /**
   *  Parse string into a number.
   *  @param str string to parse
   *  @return number or {@link #NO_NUMBER} if string does not contain a number
   */
  private static int extractNumber(String str)
  {
    try {
      return Integer.parseInt(str.trim());
    } catch (NumberFormatException x) {
      return NO_NUMBER;
    }
  }

  /**
   *  Draw all game objects.
   *  @param g graphics context
   */
  public void draw(Graphics2D g)
  {
    for (Asteroid asteroid: asteroids) {
      asteroid.draw(g);
    }
    if (ufo != null) {
      ufo.draw(g);
    }
    if (spaceShip != null) {
      spaceShip.draw(g);
    }
    for (Bullet bullet: bullets) {
      bullet.draw(g);
    }
    for (Explosion explode: explosions) {
      explode.draw(g);
    }
  }

  /**
   *  Get the frame id.
   *
   *  The frame id is a counter which is incremented by mame each time a frame is sent.
   *  @return frame id
   */
  public byte getId()
  {
    return id;
  }

  /**
   *  Get the ping associated with this frame.
   *  @return a number between <code>1</code> and <code>255</code> if there is a ping associated with this frame,
   *          otherwise <code>0</code>
   */
  public int getPing()
  {
    return ping;
  }

  /**
   *  Get the ping time associated with this frame.
   *  @return ping time or <code>0</code> if there is no ping associated with this frame
   */
  public long getPingTime()
  {
    return pingTime;
  }

  /**
   *  Get the time when this frame was received.
   *  @return receive time (compare System.currentTimeMillis())
   */
  public long getReceiveTime()
  {
    return receiveTime;
  }

  /**
   *  Get the ufo.
   *  @return ufo or <code>null</code> if no ufo is present
   */
  public Ufo getUfo()
  {
    return ufo;
  }

  /**
   *  Get the number of asteroids.
   *  @return number of asteroids
   */
  public int getAsteroidCount()
  {
    return asteroids.size();
  }

  /**
   *  Get the number of bullets.
   *  @return number of bullets
   */
  public int getBulletCount()
  {
    return bullets.size();
  }

  /**
   *  Get the number of possible targets.
   *  Targets are asteroids and ufos.
   *  @return number of targets
   */
  public int getTargetCount()
  {
    return ufo != null  ?
            getAsteroidCount() + 1  :
            getAsteroidCount();
  }

  /**
   *  Get the asteroids.
   *
   *  @return the asteroids
   */
  public Collection<Asteroid> getAsteroids()
  {
    return Collections.unmodifiableCollection(asteroids);
  }

  /**
   *  Get the space ship.
   *  @return space ship or <code>null</code> if there is no space ship present
   */
  public SpaceShip getSpaceShip()
  {
    return spaceShip;
  }

  /**
   *  Get the bullets.
   *
   *  @return bullets
   */
  public Collection<Bullet> getBullets()
  {
    return Collections.unmodifiableCollection(bullets);
  }

  /**
   *  Get the explosions.
   *  @return explosions
   */
  public Collection<Explosion> getExplosions()
  {
    return Collections.unmodifiableCollection(explosions);
  }

  /**
   *  Get the highscore.
   *  @return highscore
   */
  public int getHighscore()
  {
    return highscore;
  }

  /**
   *  Set the highscore.
   *  This may be used for overrun fixes.
   *  @param highscore new highscore
   */
  public void setHighscore(int highscore)
  {
    this.highscore = highscore;
  }

  /**
   *  Get the score.
   *  @return score
   */
  public int getScore()
  {
    return score;
  }

  /**
   *  Set the score.
   *  This may be used for overrun fixes.
   *  @param score new score
   */
  public void setScore(int score)
  {
    this.score = score;
  }

  /**
   *  Get some other number in left top.
   *  @return value of some other number in right top or {@link #NO_NUMBER} if number is not displayed
   */
  public int getSonirt()
  {
    return sonirt;
  }

  /**
   *  Is the game running.
   *
   *  This method assumes that the game is running when {@link #getSonirt()} returns a valid number.
   *  @return <code>true</code> if the game is running, otherwise <code>false</code>
   */
  public boolean isGameRunning()
  {
    return sonirt == NO_NUMBER;
  }

  /**
   *  Get the number of ship symybols in upper left corner.
   *  @return number of ships
   */
  public int getNrShips()
  {
    return nrShips;
  }

  /**
   *  Is the start text {@link #GAME_START_STRING} visible?
   *
   *  Please note that the text is blinking, so it is not visible in each screen.
   *  @return is the game start text displayed?
   */
  public boolean isGameStartDisplayed()
  {
    return gameStartDisplayed;
  }

  /**
   *  Is the game end string displayed?
   *  This indicates a finished game.
   *  @return game end displayed?
   */
  public boolean isGameEndDisplayed()
  {
    return gameEndDisplayed;
  }

  /**
   *  Is the player 1 message displayed?
   *  The indicates a starting game.
   *  @return player 1 message displayed
   */
  public boolean isPlayer1Displayed()
  {
    return player1Displayed;
  }

  /**
   *  Get the texts which are display on screen.
   *  @return texts
   */
  public Collection<Text> getTexts()
  {
    return Collections.unmodifiableCollection(texts);
  }

  /**
   *  Get the counter (index) of this frame.
   *  The counter is increased on each received frame
   *  and tags it with an unique number.
   *  @return frame counter
   */
  public int getCounter()
  {
    return counter;
  }

  /**
   *  Creates a canonized string from any string.
   *  A canonized string is one which could be displayed by the game.
   *  All letters are substituted by uppercase,
   *  and all unknown characters are substituted by space.
   *  @param str input string
   *  @return canonized string
   */
  public static String canonize(String str)
  {
    str = str.toUpperCase();
    StringBuilder result = new StringBuilder();
    for (char ch: str.toCharArray()) {
      if (ch == 'O') { // special handling of oh
        result.append('0');
      }
      else if (CHAR_MAP.values().contains(ch)) {
        result.append(ch);
      }
      else {
        result.append(' ');
      }
    }
    return result.toString();
  }

  /**
   *  Return all game objects in this frame.
   *  @return collection of game objects
   */
  public Collection<GameObject> getGameObjects()
  {
    Collection<GameObject> result = new ArrayList<GameObject>(asteroids.size() +
                                                              bullets.size() +
                                                              texts.size() +
                                                              explosions.size() +
                                                              2);
    if (spaceShip != null) {
      result.add(spaceShip);
    }
    if (ufo != null) {
      result.add(ufo);
    }
    result.addAll(asteroids);
    result.addAll(bullets);
    result.addAll(explosions);
    result.addAll(texts);
    return result;
  }

  /**
   * Get the moving game objects.
   * @return collection of moving game objects
   */
  public Collection<MovingGameObject> getMovingGameObjects()
  {
    Collection<MovingGameObject> result = new ArrayList<MovingGameObject>(asteroids.size() +
                                                                        bullets.size() +
                                                                        explosions.size() +
                                                                        2);
    if (spaceShip != null) {
      result.add(spaceShip);
    }
    if (ufo != null) {
      result.add(ufo);
    }
    result.addAll(asteroids);
    result.addAll(bullets);
    result.addAll(explosions);
    return result;
  }

  /**
   *  Return a collection of game objects which overlap with a given rectangle.
   *  @param hitRect hit rectangle
   *  @return list of overlapping objects
   */
  public Collection<GameObject> getOverlappingObjects(Rectangle hitRect)
  {
    Collection<GameObject> result = new LinkedList<GameObject>();
    for (GameObject go: getGameObjects()) {
      if (go.isOverlappingWith(hitRect)) {
        result.add(go);
      }
    }
    return result;
  }

  /**
   * Get the properties of this object.
   *
   * @return collection of properties
   */
  public Collection<Property> getProperties()
  {
    Collection<Property> props = new LinkedList<Property>();
    props.add(new Property<String>("Type", "Frame"));
    props.add(new Property<Integer>("Index", getCounter()));
    props.add(new Property<Integer>("MAME ID", Tools.byteToUnsigned(id)));
    props.add(new Property<Integer>("Ping ID", Tools.byteToUnsigned(ping)));
    props.add(new Property<Long>("Receive Time", receiveTime));
    props.add(new Property<Long>("Ping Time", pingTime));
    props.add(new Property<Integer>("Score", score));
    props.add(new Property<Integer>("High Score", highscore));
    props.add(new Property<Integer>("Some Other Number In Right Top", sonirt));
    props.add(new Property<Boolean>("'Game Start' Displayed?", gameStartDisplayed));
    props.add(new Property<Boolean>("'Player 1' Displayed?", player1Displayed));
    props.add(new Property<Boolean>("'Game End' Displayed?", gameEndDisplayed));
    props.add(new Property<Integer>("Number of Ships in Reserve", nrShips));
    props.add(new Property<Integer>("Number of Asteroids", asteroids.size()));
    props.add(new Property<Integer>("Number of Bullets", bullets.size()));
    props.add(new Property<Integer>("Number of Explosions", explosions.size()));
    props.add(new Property<Integer>("Number of Texts", texts.size()));
    props.add(new Property<Boolean>("Space Ship displayed?", spaceShip != null));
    props.add(new Property<Boolean>("Ufo displayed?", ufo != null));
    return props;
  }
}