// ============================================================================
// File:               $File$
//
// Project:            
//
// Purpose:            
//
// Author:             Rammi
//                     This code is in the public domain.
//                     Use at own risk.
//                     No guarantees given.
//
// Copyright Notice:   (c) 2008  Rammi (rammi@caff.de)
//
// Latest change:      $Date$
//
// History:	       $Log$
//=============================================================================
package de.caff.asteroid.analysis;

import de.caff.asteroid.*;
import de.caff.util.Tools;

import java.util.*;
import java.util.zip.ZipFile;
import java.util.zip.ZipEntry;
import java.io.*;

/**
 *  Read a dump written by a {@link de.caff.asteroid.analysis.DatagramDumper}.
 *
 *  This class is part of a solution for a
 *  <a href="http://www.heise.de/ct/creativ/08/02/details/">competition by the German computer magazine c't</a>.
 */
public class DumpFile
        implements FileFormat
{
  /** Mark type set between in-game and between-game frames. */
  public enum MarkType
  {
    GAME_START, // in-game start
    GAME_END    // between-game start
  }

  /**
   *  Marks are set to the first frame in a game and the first game after the game ended.
   */
  public static class Mark
  {
    /** Index of frame. */
    private int frameIndex;
    /** Mark of frame. */
    private MarkType markType;

    /**
     *  Constructor.
     *  @param frameIndex index of frame
     *  @param markType     mark type
     */
    public Mark(int frameIndex, MarkType markType)
    {
      this.frameIndex = frameIndex;
      this.markType = markType;
    }

    /**
     *  Get the frame associated wiht
     *  @return frame index of associated frame
     */
    public int getFrameIndex()
    {
      return frameIndex;
    }

    /**
     *  Get the mark type.
     *  @return mark type
     */
    public MarkType getMarkType()
    {
      return markType;
    }
  }
  /** The list of frames. */
  private List<FrameKeyInfo> infos = new ArrayList<FrameKeyInfo>();
  /** The list of marks. */
  private List<Mark> marks = new ArrayList<Mark>();
  /** The list of markers. */
  /** Exception if reading was stopped by exception. */
  private IOException finalException;
  /** The list of change listeners. */
  private List<DumpFileChangeListener> changeListeners = new LinkedList<DumpFileChangeListener>();

  /**
   *  Constructor.
   *  @param filename dump file path
   *  @throws IOException if early i/o errors (file not found or similar) occur
   */
  public DumpFile(String filename)
          throws IOException
  {
    BufferedInputStream is = null;
    if (filename.toLowerCase().endsWith(".zip")) {
      try {
        ZipFile zip = new ZipFile(filename);
        for (Enumeration<? extends ZipEntry> it = zip.entries();  it.hasMoreElements(); ) {
          ZipEntry entry = it.nextElement();
          is = new BufferedInputStream(zip.getInputStream(entry));
          break;
        }
      } catch (IOException e) {
        e.printStackTrace();
      }
    }
    
    if (is == null) {
      FileInputStream fis = new FileInputStream(filename);
      is = new BufferedInputStream(fis);
    }
    byte[] intro = new byte[INTRO.length];
    if (is.read(intro) < intro.length) {
      throw new IOException(String.format("%s: too short!", filename));
    }
    for (int i = 0;  i < intro.length;  ++i) {
      if (intro[i] != INTRO[i]) {
        throw new IOException(String.format("%s: intro error at %d!", filename, i));
      }
    }
    // from now on i/o errors are accepted.
    byte[] frame = new byte[MAME_DATAGRAM_SIZE];
    long[] pingTimes = new long[256];
    MarkType markType = null;
    try {
      int nextByte;
      FrameKeyInfo info = null;
      long timestamp;
      while ((nextByte = is.read()) >= 0) {
        int bytesRead;
        switch (nextByte) {
        case OUTGOING_MARKER:
          timestamp = readTimestamp(is);
          bytesRead = is.read(frame, 0, KEY_PACKET_SIZE);
          if (bytesRead < KEY_PACKET_SIZE) {
            throw new IOException(String.format("%s: premature EOF!", filename));
          }
          checkKeyPacket(frame);
          if (info != null) {
            info.addButtons(timestamp, frame[KEY_MASK_INDEX], frame[KEY_PING_INDEX]);
          }
          else {
            System.out.println(String.format("%s: keys before first frame: %s!", filename, new Buttons(frame[KEY_MASK_INDEX])));
          }
          pingTimes[Tools.byteToUnsigned(frame[KEY_PING_INDEX])] = timestamp;
          break;

        case INCOMING_MARKER:
          timestamp = readTimestamp(is);
          bytesRead = is.read(frame);
          if (bytesRead < MAME_DATAGRAM_SIZE) {
            throw new IOException(String.format("%s: premature EOF!", filename));
          }
          info = new FrameKeyInfo(timestamp, frame, pingTimes);
          MarkType newMarkType = getFrameType(info);
          if (newMarkType != markType) {
            marks.add(new Mark(infos.size(), newMarkType));
            markType = newMarkType;
          }
          infos.add(info);
          break;
        }
      }
    } catch (IOException e) {
      finalException = e;
      e.printStackTrace();
    }
    runPreparer(new ScoreFixer());
  }

  /**
   *  Read a timestamp.
   *  @param is  input stream
   *  @return time stamp
   *  @throws IOException if not enough bytes are available
   */
  private static long readTimestamp(InputStream is) throws IOException
  {
    long result = 0;
    for (int b = 0;  b < Long.SIZE/Byte.SIZE;  ++b) {
      int bb = is.read();
      if (bb < 0) {
        throw new IOException("Premature EOF!");
      }
      result |= ((long)(bb & 0xFF)) << Byte.SIZE*b; 
    }
    return result;
  }

  /**
   * Check a key packet for correct format.
   * @param keyPacket   key packet to check
   * @throws IOException if format is not correct
   */
  private static void checkKeyPacket(byte[] keyPacket) throws IOException
  {
    for (int i = KEY_PACKET_INTRO.length - 1;  i >= 0;  --i) {
      if (keyPacket[i] != KEY_PACKET_INTRO[i]) {
        throw new IOException("Invalid key packet!");
      }
    }
  }

  /**
   *  Give a frame a marker type.
   *  @param info frame key info
   *  @return a marker
   */
  private static MarkType getFrameType(FrameKeyInfo info)
  {
    return info.getFrameInfo().isGameRunning() ? MarkType.GAME_START : MarkType.GAME_END;
  }

  /**
   *  Get the list of frame key infos.
   *  @return list of frame key infos
   */
  public List<FrameKeyInfo> getInfos()
  {
    return infos;
  }

  /**
   *  Get the last frame key info.
   *  @return last info or <code>null</code> if there are no infos
   */
  public FrameKeyInfo getLastInfo()
  {
    return infos.isEmpty() ? null : infos.get(infos.size() - 1);
  }

  /**
   *  Get the list of markers.
   *  @return marker list
   */
  public List<Mark> getMarks()
  {
    return marks;
  }

  /**
   *  Get the exception which ended the input of the file (if any).
   *  @return final exception or <code>null</code> if file reading ended gracefully
   */
  public IOException getFinalException()
  {
    return finalException;
  }

  /**
   *  Run a preparer on the frames.
   *  The preparer is run in a background thread
   *  @param preparer preparer to run
   */
  public void runPreparer(final FramePreparer preparer)
  {
    new Thread(new Runnable()
    {
      public void run()
      {
        runPrep(preparer);
      }
    }).start();
  }

  /**
   *  Run a preparer on the frames (internally).
   *  @param preparer preparer to run
   */
  private synchronized void runPrep(FramePreparer preparer)
  {
    LinkedList<FrameInfo> pending = new LinkedList<FrameInfo>();
    for (FrameKeyInfo info: infos) {
      pending.add(info.getFrameInfo());
      if (pending.size() > Communication.MAX_FRAMES_KEPT) {
        pending.remove(0);
      }
      preparer.prepareFrames(pending);
    }
    informDumpFileChangeListeners();
  }

  /**
   *  Add a dump file change listener called if the dumpe file content changes.
   *  @param listener listener to add
   */
  public void addDumpFileChangeListener(DumpFileChangeListener listener)
  {
    synchronized (changeListeners) {
      changeListeners.add(listener);
    }
  }

  /**
   *  Remove a dump file change listener.
   *  @param listener listener to remove
   *  @return <code>true</code> if the listener was removed<br>
   *          <code>false</code> otherwise
   */
  public boolean removeDumpFileChangeListener(DumpFileChangeListener listener)
  {
    synchronized (changeListeners) {
      return changeListeners.remove(listener);
    }
  }

  /**
   *  Inform all registered dump file change listeners.
   */
  private void informDumpFileChangeListeners()
  {
    Collection<DumpFileChangeListener> tmp;
    synchronized (changeListeners) {
      if (changeListeners.isEmpty()) {
        return;
      }
      tmp = new ArrayList<DumpFileChangeListener>(changeListeners);
    }
    for (DumpFileChangeListener listener: tmp) {
      listener.dumpFileChange();
    }
  }

  /**
   *  Test code.
   *  @param args call with dump file arguments
   *  @throws IOException on early read errors
   */
  public static void main(String[] args)
          throws IOException
  {
    for (String arg: args) {
      DumpFile file = new DumpFile(arg);
      System.out.println(String.format("%s: %d frames and %d marks.", arg, file.getInfos().size(), file.getMarks().size()));
      file.runPrep(new ScoreFixer());
      if (!file.getInfos().isEmpty()) {
        System.out.println("Score at end: "+file.getLastInfo().getFrameInfo().getScore());
      }
    }
  }
}
