// ============================================================================
// File:               $File$
//
// Project:            
//
// Purpose:            
//
// Author:             Rammi
//
// Copyright Notice:   (c) 2008  Rammi (rammi@caff.de)
//                     This code is in the public domain.
//                     Use at own risk.
//                     No guarantees given.
//
// Latest change:      $Date$
//
// History:	       $Log$
//=============================================================================
package de.caff.asteroid.analysis;

import de.caff.asteroid.*;

import javax.swing.*;
import javax.swing.event.TreeSelectionListener;
import javax.swing.event.TreeSelectionEvent;
import javax.swing.table.DefaultTableModel;
import javax.swing.tree.*;
import java.util.Collection;
import java.util.LinkedList;
import java.awt.*;
import java.awt.event.MouseWheelListener;
import java.awt.event.MouseWheelEvent;

/**
 *  Display information about the current frame.
 *
 *  This class is part of a solution for a
 *  <a href="http://www.heise.de/ct/creativ/08/02/details/">competition by the German computer magazine c't</a>.
 */
public class FrameKeyInfoDisplay
        extends JPanel
{
  /** Property table columns. */
  private static final String[] COLUMNS = { "Name", "Value" };
  /** Null value for selection are changes. */
  private static final Rectangle[] NO_BOUNDS = new Rectangle[0];

  /** Tree of frame entries. */
  private JTree tree;
  /** The table where the properties are displayed. */
  private JTable propertyTable;
  /** The listeners for selection area changes. */
  private Collection<SelectionAreaListener> selectionAreaListeners = new LinkedList<SelectionAreaListener>();

  /**
   *  Constructor.
   *  @param timeLine time line
   *  @param velocityPreparer preparer used to calculate velocities (and more)
   *  @param provider         additional provider of drawable information or <code>null</code>
   *  @param simple           use simple display without information panels? 
   */
  public FrameKeyInfoDisplay(final TimeLine timeLine,
                             FramePreparer velocityPreparer,
                             DrawableProvider provider,
                             boolean simple)
  {
    setLayout(new BorderLayout());

    EnhancedFrameDisplay frameDisplay = new EnhancedFrameDisplay(timeLine, this);
    frameDisplay.setDrawableProvider(provider);
    timeLine.addFrameListener(frameDisplay);
    if (timeLine.hasFrames()) {
      frameDisplay.frameReceived(timeLine.getCurrentInfo().getFrameInfo());
    }
    timeLine.addDumpFileChangeListener(frameDisplay);

    if (!simple) {
      tree = new JTree();
      propertyTable = new JTable();
      JScrollPane treePane = new JScrollPane(tree);
      JScrollPane tablePane = new JScrollPane(propertyTable);
      treePane.setBorder(BorderFactory.createTitledBorder("Game Objects"));
      tablePane.setBorder(BorderFactory.createTitledBorder("Properties of Selected Object"));

      timeLine.addFrameListener(new FrameListener() {
        public void frameReceived(FrameInfo frame)
        {
          setTreeFrom(timeLine.getCurrentInfo());
        }
      });

      tree.addTreeSelectionListener(new TreeSelectionListener()
      {
        public void valueChanged(TreeSelectionEvent e)
        {
          TreePath[] paths = tree.getSelectionPaths();
          Rectangle[] bounds;
          if (paths != null) {
            if (paths.length == 1) {
              setTableFrom(((DefaultMutableTreeNode)paths[0].getLastPathComponent()).getUserObject());
            }
            else {
              setTableFrom(null);
            }
            Collection<Rectangle> boundList = new LinkedList<Rectangle>();
            for (int p = paths.length - 1;  p >= 0;  --p) {
              Object obj = ((DefaultMutableTreeNode)paths[p].getLastPathComponent()).getUserObject();
              try {
                boundList.add(((GameObject)obj).getBounds());
              } catch (ClassCastException x) {
                // considered okay, happens on frame node
              }
            }
            bounds = boundList.toArray(new Rectangle[boundList.size()]);
          }
          else {
            setTableFrom(null);
            bounds = NO_BOUNDS;
          }
          informSelectionAreaListeners(bounds);
        }
      });
      addSelectionAreaListener(frameDisplay);

      JSplitPane splitter = new JSplitPane(JSplitPane.VERTICAL_SPLIT);
      splitter.setTopComponent(new KeyInfoDisplay(timeLine));
      splitter.setBottomComponent(tablePane);

      JPanel leftPane = new JPanel(new BorderLayout());
      leftPane.add(new ButtonBar(timeLine, velocityPreparer), BorderLayout.NORTH);
      leftPane.add(splitter, BorderLayout.CENTER);
      leftPane.add(frameDisplay, BorderLayout.SOUTH);

      add(treePane, BorderLayout.CENTER);
      add(leftPane, BorderLayout.EAST);


      setTreeFrom(timeLine.getCurrentInfo());

      setMinimumSize(new Dimension(600, 300));
      setPreferredSize(new Dimension(800, 400));
    }
    else {
      add(new ButtonBar(timeLine, velocityPreparer), BorderLayout.NORTH);
      add(frameDisplay, BorderLayout.CENTER);
    }

    frameDisplay.addMouseWheelListener(timeLine.getMouseWheelListener());
  }

  /**
   *  Set the tree from a new frame info.
   *  @param info new info
   */
  private void setTreeFrom(FrameKeyInfo info)
  {
    if (tree != null) {
      if (info != null) {
        DefaultMutableTreeNode frameNode = new DefaultMutableTreeNode(info.getFrameInfo());
        for (GameObject obj: info.getFrameInfo().getGameObjects()) {
          frameNode.add(new DefaultMutableTreeNode(obj));
        }
        tree.setModel(new DefaultTreeModel(frameNode));
        tree.setSelectionRow(0);
      }
      else {
        tree.setModel(new DefaultTreeModel(null));
      }
    }
  }

  /**
   *  Set the table content from a new selected object.
   *  @param userObj selected object in tree
   */
  private void setTableFrom(Object userObj)
  {
    if (propertyTable != null) {
      Object[][] data;
      if (userObj != null  &&  userObj instanceof PropertyProvider) {
        Collection<Property> props = ((PropertyProvider)userObj).getProperties();
        data = new Object[props.size()][2];
        int row = 0;
        for (Property p: props) {
          data[row][0] = p.getName();
          data[row][1] = p.getValue();
          ++row;
        }
      }
      else {
        data = new Object[0][2];
      }
      propertyTable.setModel(new DefaultTableModel(data, COLUMNS));
    }
  }

  /**
   * Add a selection area change listener.
   * It will be called if the selection changes.
   * @param listener listener to add
   */
  public void addSelectionAreaListener(SelectionAreaListener listener)
  {
    selectionAreaListeners.add(listener);
  }

  /**
   *  Remove a selection area change listener.
   *  @param listener listener to remove
   *  @return <code>true</code> if the listener was removed<br>
   *          <code>false</code> otherwise
   */
  public boolean removeSelectionAreaListener(SelectionAreaListener listener)
  {
    return selectionAreaListeners.remove(listener);
  }

  /**
   *  Call all selection area change listeners.
   *  @param selections new selected areas
   */
  protected void informSelectionAreaListeners(Rectangle[] selections)
  {
    for (SelectionAreaListener listener: selectionAreaListeners) {
      listener.selectionAreasChanged(selections);
    }
  }

  /**
   *  Select the given user objects.
   *  @param userObjects user objects to select
   */
  public void setSelectedUserObjects(Object[] userObjects)
  {
    if (tree != null) {
      TreePath[] selectionPaths;
      if (userObjects.length > 0) {
        DefaultMutableTreeNode root = (DefaultMutableTreeNode)tree.getModel().getRoot();
        Collection<TreePath> selection = new LinkedList<TreePath>();
        collectSelectionPaths(root, selection, userObjects);
        selectionPaths = selection.toArray(new TreePath[selection.size()]);
      }
      else {
        selectionPaths = new TreePath[0];
      }
      tree.setSelectionPaths(selectionPaths);
    }
  }

  /**
   *  Helper method used in {@link #setSelectedUserObjects(Object[])}.
   *  @param node        current node
   *  @param selection   collection for collecting selection paths
   *  @param userObjects user objects to select
   */
  private static void collectSelectionPaths(DefaultMutableTreeNode node, Collection<TreePath> selection, Object[] userObjects)
  {
    for (Object o: userObjects) {
      if (node.getUserObject() == o) {
        selection.add(new TreePath(node.getPath()));
        break;
      }
    }
    for (int ch = 0;  ch < node.getChildCount(); ++ch) {
      collectSelectionPaths(((DefaultMutableTreeNode)node.getChildAt(ch)), selection, userObjects);
    }
  }

}
