// ============================================================================
// File:               $File$
//
// Project:            
//
// Purpose:            
//
// Author:             Rammi
//
// Copyright Notice:   (c) 2008  Rammi (rammi@caff.de)
//                     This code is in the public domain.
//                     Use at own risk.
//                     No guarantees given.
//
// Latest change:      $Date$
//
// History:	       $Log$
//=============================================================================
package de.caff.asteroid.analysis;

import de.caff.asteroid.FrameDisplay;
import de.caff.asteroid.GameObject;
import de.caff.asteroid.FrameInfo;

import java.awt.*;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.geom.AffineTransform;
import java.awt.geom.GeneralPath;
import java.util.Collection;

/**
 *  Frame display with additional selection.
 *
 *  This class is part of a solution for a
 *  <a href="http://www.heise.de/ct/creativ/08/02/details/">competition by the German computer magazine c't</a>.
 */
public class EnhancedFrameDisplay
        extends FrameDisplay
        implements SelectionAreaListener,
                   DumpFileChangeListener
{

  private static final Color SELECTION_COLOR = Color.magenta;

  private Rectangle[] selections = new Rectangle[0];
  private TimeLine timeLine;

  /**
   * Constructor.
   * @param timeLine time line
   */
  public EnhancedFrameDisplay(TimeLine timeLine)
  {
    this(0, timeLine, null);
  }

  /**
   * Constructor.
   * @param timeLine time line
   * @param fid      displayer of infos of clicked objects
   */
  public EnhancedFrameDisplay(TimeLine timeLine, final FrameKeyInfoDisplay fid)
  {
    this(0, timeLine, fid);
  }

  /**
   * Constructor.
   * @param width    the fiy width of this component or <code>null</code> for a variable size
   * @param timeLine time line
   * @param fid      displayer of infos of clicked objects
   */
  public EnhancedFrameDisplay(int width, TimeLine timeLine, final FrameKeyInfoDisplay fid)
  {
    super(width);
    this.timeLine = timeLine;
    if (fid != null) {
      addMouseListener(new MouseAdapter() {
        /**
         * {@inheritDoc}
         */
        @Override
        public void mouseClicked(MouseEvent e)
        {
          if (e.getButton() == MouseEvent.BUTTON1) {
            Collection<GameObject> gameObjects = pickAt(e.getPoint(), 2);
            fid.setSelectedUserObjects(gameObjects.toArray(new GameObject[gameObjects.size()]));
          }
        }
      });
    }
  }

  /**
   * Invoked by Swing to draw components.
   * Applications should not invoke <code>paint</code> directly,
   * but should instead use the <code>repaint</code> method to
   * schedule the component for redrawing.
   * <p/>
   * This method actually delegates the work of painting to three
   * protected methods: <code>paintComponent</code>,
   * <code>paintBorder</code>,
   * and <code>paintChildren</code>.  They're called in the order
   * listed to ensure that children appear on top of component itself.
   * Generally speaking, the component and its children should not
   * paint in the insets area allocated to the border. Subclasses can
   * just override this method, as always.  A subclass that just
   * wants to specialize the UI (look and feel) delegate's
   * <code>paint</code> method should just override
   * <code>paintComponent</code>.
   *
   * @param g the <code>Graphics</code> context in which to paint
   * @see #paintComponent
   * @see #paintBorder
   * @see #paintChildren
   * @see #getComponentGraphics
   * @see #repaint
   */
  @Override
  public void paint(Graphics g)
  {
    super.paint(g);
    Graphics2D g2 = (Graphics2D)g.create();
    g2.setStroke(new BasicStroke(2));
    g2.setColor(SELECTION_COLOR);
    AffineTransform trafo = getTrafo();
    for (Rectangle rect: selections) {
      GeneralPath p = new GeneralPath();
      p.moveTo((float)rect.getMinX(), (float)rect.getMinY());
      p.lineTo((float)rect.getMaxX(), (float)rect.getMinY());
      p.lineTo((float)rect.getMaxX(), (float)rect.getMaxY());
      p.lineTo((float)rect.getMinX(), (float)rect.getMaxY());
      p.closePath();
      p.transform(trafo);
      g2.draw(p);
    }

  }

  /**
   *  Called if selection areas have changed.
   *  @param areas new areas
   */
  public void selectionAreasChanged(Rectangle[] areas)
  {
    selections = areas;
    repaint();
  }

  /**
   * Called if the dump file has (possibly) changed.
   */
  public void dumpFileChange()
  {
    repaint();
  }

  /**
   * Get the duration since the last game start.
   *
   * @param frame frame for which the duration is requested
   * @return time since the current game started in seconds
   */
  @Override
  protected int getSessionTime(FrameInfo frame)
  {
    return timeLine.getSessionTime(frame.getIndex())/1000;
  }
}
