// ============================================================================
// File:               $File$
//
// Project:            
//
// Purpose:            
//
// Author:             Rammi
//
// Copyright Notice:   (c) 2008  Rammi (rammi@caff.de)
//                     This code is in the public domain.
//                     Use at own risk.
//                     No guarantees given.
//
// Latest change:      $Date$
//
// History:	       $Log$
//=============================================================================
package de.caff.asteroid;

import javax.swing.*;
import java.awt.*;
import java.awt.geom.AffineTransform;
import java.awt.geom.NoninvertibleTransformException;
import java.awt.geom.Point2D;
import java.util.Collection;
import java.util.Collections;

/**
 *  Component to display a frame as received from MAME.
 *
 *  Because this an example of a frame listener note the usage of synchronized access to
 *  the frameInfo object passed between threads and the usage of repaint() to inform the AWT
 *  thread to start drawing.
 *
 *  This class is part of a solution for a
 *  <a href="http://www.heise.de/ct/creativ/08/02/details/">competition by the German computer magazine c't</a>
 */
public class FrameDisplay
  extends JComponent
  implements FrameListener,
             GameData
{
  /** Color used for game duration. */
  private static final Color TIME_COLOR = new Color(0x00, 0xFF, 0x00, 0x80);
  /** Color used for score. */
  private static final Color SCORE_COLOR = new Color(0xFF, 0xFF, 0xFF, 0x80);
  /** The position where to display the pending ships (lives). */
  private static final Point PENDING_SHIPS_POSITION = new Point(160, 852);
  /** The delta for pending ships display. */
  private static final int PENDING_SHIPS_DELTA_X = 20;

  /** The basic width (if 0 then adapt). */
  private int baseWidth;
  /** The currently displayed frame info. */
  private FrameInfo frameInfo;
  /** The transformation from frame space into component space. */
  private AffineTransform trafo;
  /** Provider for additional drawables in MAME space. */
  private DrawableProvider drawableProvider;
  /** Start frame. */
  private int startFrame = -1;

  /**
   *  Constructor.
   *
   *  The component is always constructed with a aspect ratio of 4:3.
   *  @param width width to use for this component
   */
  public FrameDisplay(int width)
  {
    baseWidth = width;
    if (width > 0) {
      Dimension size = new Dimension(width, 3*width/4);
      setMaximumSize(size);
      setMinimumSize(size);
      setPreferredSize(size);
      setSize(size);
      calculateScaling();
    }
    else {
      setMinimumSize(new Dimension(200, 150));
      setPreferredSize(new Dimension(640, 480));
    }
    setOpaque(true);
    setDoubleBuffered(true);
  }

  /**
   *  Calculate the scaling from the current size.
   */
  private void calculateScaling()
  {
    int width = baseWidth <= 0  ?
            Math.min(getWidth(), 4*getHeight()/3) :
            baseWidth;
    double scaling = width/(double)EXTENT;
    // NOTE: take care of y pointing upwards in Asteroids, but downwards on screen
    trafo = AffineTransform.getTranslateInstance(0, MIN_Y-EXTENT);
    trafo.preConcatenate(AffineTransform.getScaleInstance(scaling, -scaling));
  }

  /**
   *  Get the correct size keeping the aspect ratio.
   *  @return correct size
   */
  private Dimension getCorrectSize()
  {
    int width = baseWidth <= 0  ?
            Math.min(getWidth(), 4*getHeight()/3) :
            baseWidth;
    return new Dimension(width, 3*width/4);
  }

  /**
   * Invoked by Swing to draw components.
   * Applications should not invoke <code>paint</code> directly,
   * but should instead use the <code>repaint</code> method to
   * schedule the component for redrawing.
   * <p/>
   * This method actually delegates the work of painting to three
   * protected methods: <code>paintComponent</code>,
   * <code>paintBorder</code>,
   * and <code>paintChildren</code>.  They're called in the order
   * listed to ensure that children appear on top of component itself.
   * Generally speaking, the component and its children should not
   * paint in the insets area allocated to the border. Subclasses can
   * just override this method, as always.  A subclass that just
   * wants to specialize the UI (look and feel) delegate's
   * <code>paint</code> method should just override
   * <code>paintComponent</code>.
   *
   * @param g the <code>Graphics</code> context in which to paint
   * @see #paintComponent
   * @see #paintBorder
   * @see #paintChildren
   * @see #getComponentGraphics
   * @see #repaint
   */
  @Override
  public void paint(Graphics g)
  {
    if (baseWidth == 0) {
      calculateScaling();
    }
    g.setColor(Color.black);
    Dimension size = getCorrectSize();
    g.fillRect(0, 0, size.width, size.height);
    g.setClip(0, 0, size.width, size.height);
    FrameInfo info;
    synchronized (this) {
      info = frameInfo;
    }
    if (info != null) {
      Graphics2D g2 = (Graphics2D)g.create();
      Text.setFontRenderContext(((Graphics2D)g).getFontRenderContext());
      g2.transform(trafo);
      info.draw(g2);
      if (drawableProvider != null) {
        drawableProvider.draw(g2, info);
      }
      // SHIPs
      g2.setColor(Color.gray);
      for (int s = info.getNrShips() - 1;  s >= 0;  --s) {
        g2.draw(SpaceShip.getShipShape(PENDING_SHIPS_POSITION.x + s * PENDING_SHIPS_DELTA_X,
                                       PENDING_SHIPS_POSITION.y,
                                       Math.PI/2));
      }
      Point pos = new Point();
      if (!info.isGameRunning()) {
        g.setColor(new Color(0xFF, 0xFF, 0x00, 0x80));
        Point2D pos0 = new Point2D.Double(0, 0);
        Point2D pos1 = new Point2D.Double(0, 1);
        trafo.transform(pos0, pos0);
        trafo.transform(pos1, pos1);
        float fontScale = (float)Math.abs(pos0.getY() - pos1.getY());
        for (Text txt: info.getTexts()) {
          trafo.transform(txt.getLocation(), pos);
          // fix font size
          Font font = txt.getFont();
          g.setFont(font.deriveFont(fontScale * font.getSize2D()));
          g.drawString(txt.getText(), pos.x, pos.y);
        }
      }
      // Score
      g.setColor(SCORE_COLOR);
      trafo.transform(FrameInfo.SCORE_LOCATION_GAME, pos);
      g.drawString(String.format("%7d", info.getScore()),
                   pos.x, pos.y);

      // Duration
      int duration = getSessionTime(info);
      g.setColor(TIME_COLOR);
      g.drawString(String.format("%02d:%02d:%02d",
                                 (duration/60/60) % 100,
                                 (duration/60) % 60,
                                 duration % 60),
                   size.width - 100, 20);
    }
  }

  /**
   *  Get the duration since the last game start.
   *  @param frame frame for which the duration is requested
   *  @return time since the current game started in seconds
   */
  protected int getSessionTime(FrameInfo frame)
  {
    if (frame.isGameRunning()) {
      if (startFrame == -1L) {
        startFrame = frame.getIndex();
      }
      return (frame.getIndex() - startFrame) / FRAMES_PER_SECOND;
    }
    else {
      startFrame = frame.getIndex();
    }
    return 0;
  }

  /**
   *  Called each time a frame is received.
   *
   *  <b>ATTENTION:</b> this is called from the communication thread!
   *  Implementing classes must be aware of this and take care by synchronization or similar!
   *  @param frame the received frame
   */
  public void frameReceived(FrameInfo frame)
  {
    synchronized (this) {
      frameInfo = frame;
    }
    repaint();
  }

  /**
   * Return objects at a given position.
   * @param p          pick position
   * @param pickRadius pick radius
   * @return collection of game objects found at the given position
   */
  public Collection<GameObject> pickAt(Point p, int pickRadius)
  {
    Point min = new Point(p.x - pickRadius, p.y + pickRadius); // using knowledge of negative y scaling
    Point max = new Point(p.x + pickRadius, p.y - pickRadius);
    try {
      AffineTransform inverse = new AffineTransform(trafo).createInverse();
      inverse.transform(min, min);
      inverse.transform(max, max);
      Rectangle hitRect = new Rectangle(min.x, min.y, max.x - min.x, max.y - min.y);
      synchronized (this) {
        return frameInfo.getOverlappingObjects(hitRect);
      }
    } catch (NoninvertibleTransformException e) {
      e.printStackTrace();
    }
    return Collections.emptyList();
  }

  /**
   *  Get the displayed frame info.
   *  @return frame info
   */
  public FrameInfo getFrameInfo()
  {
    return frameInfo;
  }

  /**
   *  Get the transformation from MAME space to screen.
   *  @return transformation
   */
  protected AffineTransform getTrafo()
  {
    return trafo;
  }

  public DrawableProvider getDrawableProvider()
  {
    return drawableProvider;
  }

  public void setDrawableProvider(DrawableProvider drawableProvider)
  {
    this.drawableProvider = drawableProvider;
  }
}
