// ============================================================================
// File:               $File$
//
// Project:            
//
// Purpose:            
//
// Author:             Rammi
//
// Copyright Notice:   (c) 2008  Rammi (rammi@caff.de)
//                     This code is in the public domain.
//                     Use at own risk.
//                     No guarantees given.
//
// Latest change:      $Date$
//
// History:	       $Log$
//=============================================================================
package de.caff.asteroid;

import java.awt.*;
import java.util.Collection;
import java.util.LinkedList;

/**
 *  Generic game object in Asteroid game.
 *
 *  This class is part of a solution for a
 *  <a href="http://www.heise.de/ct/creativ/08/02/details/">competition by the German computer magazine c't</a>
 */
public abstract class GameObject
        implements GameData,
                   PropertyProvider
{
  /** The x coordinate. */
  protected final int x;
  /** The y coordinate (y is point upwards). */
  protected final int y;

  /**
   *  Constructor.
   *  @param x x coordinate
   *  @param y y coordinate (points upwards in game)
   */
  public GameObject(int x, int y)
  {
    this.x = x;
    this.y = y;
  }

  /**
   *  Get the x coordinate of the objects center.
   *  @return center x
   */
  public int getX()
  {
    return x;
  }

  /**
   *  Get the y coordinate of the objects center.
   *  @return center y
   */
  public int getY()
  {
    return y;
  }

  /**
   *  Get the squared length of the shortest vector from this to another game object.
   *
   *  Because of the torus topology it may be shorter going across the borders.
   *  @param obj other object
   *  @return squared length of shortest vector from this to the other object
   */
  public int getSquaredDistance(GameObject obj)
  {
    return getSquaredDistance(obj.getX(), obj.getY());
  }

  /**
   *  Get the squared length of the shortest vector from this objects position to another position.
   *
   *  Because of the torus topology it may be shorter going across the borders.
   *  @param px other position's x
   *  @param py other position's y
   *  @return squared length of shortest vector from this object to the other position
   */
  public int getSquaredDistance(int px, int py)
  {
    int dx = px - x;
    if (dx < -EXTENT_X/2) {
      dx += EXTENT_X;
    }
    else if (dx >= EXTENT_X/2) {
      dx -= EXTENT_X;
    }
    int dy = py - y;
    if (dy < -EXTENT_Y/2) {
      dy += EXTENT_Y;
    }
    else if (dy >= EXTENT_Y/2) {
      dy -= EXTENT_Y;
    }
    return dx*dx + dy*dy;
  }

  /**
   *  Get the squared length of the shortest vector from this objects position to another position.
   *
   *  Because of the torus topology it may be shorter going across the borders.
   *  @param px other position's x
   *  @param py other position's y
   *  @return squared length of shortest vector from this object to the other position
   */
  public double getSquaredDistance(double px, double py)
  {
    double dx = normalizeDeltaX(px - x);
    double dy = normalizeDeltaY(py - y);
    return dx*dx + dy*dy;
  }

  /**
   *  Normalize a component.
   *  @param v      component
   *  @param extent extent
   *  @return a value mapped to <code>-extent/2 &lt;= value &lt;extent/2</code>
   */
  private static int normalizeDelta(int v, int extent)
  {
    while (v < -extent/2) {
      v += extent;
    }
    while (v >= extent/2) {
      v -= extent;
    }
    return v;
  }

  /**
   *  Normalize a x vector component.
   *  @param x component
   *  @return restricted value
   */
  public static int normalizeDeltaX(int x)
  {
    return normalizeDelta(x, EXTENT_X);
  }

  /**
   *  Normalize a y vector component.
   *  @param y component
   *  @return restricted value
   */
  public static int normalizeDeltaY(int y)
  {
    return normalizeDelta(y, EXTENT_Y);
  }

  /**
   *  Normalize a component.
   *  @param v      component
   *  @param extent extent
   *  @return a value mapped to <code>-extent/2 &lt;= value &lt;extent/2</code>
   */
  private static double normalizeDelta(double v, double extent)
  {
    while (v < -extent/2) {
      v += extent;
    }
    while (v >= extent/2) {
      v -= extent;
    }
    return v;
  }

  /**
   *  Normalize a x vector component.
   *  @param x component
   *  @return restricted value
   */
  public static double normalizeDeltaX(double x)
  {
    return normalizeDelta(x, EXTENT_X);
  }

  /**
   *  Normalize a y vector component.
   *  @param y component
   *  @return restricted value
   */
  public static double normalizeDeltaY(double y)
  {
    return normalizeDelta(y, EXTENT_Y);
  }

  /**
   *  Is this object overlapping with a given rectangle?
   *  @param rect rectangle
   *  @return the answer
   */
  public boolean isOverlappingWith(Rectangle rect)
  {
    Rectangle bounds = getBounds();
    return rect.intersects(bounds);
  }

  /**
   *  Get the bounding box of this rectangle.
   *  @return the bounding box
   */
  public abstract Rectangle getBounds();

  /**
   *  Get the properties of this object.
   *  @return collection of properties
   */
  public Collection<Property> getProperties()
  {
    Collection<Property> props = new LinkedList<Property>(); // NOTE: linked list so sub classes can easily add
    props.add(new Property<String>("Type", getObjectType()));
    props.add(new Property<Point>("Position", getLocation()));
    return props;
  }

  /**
   *  Get the location of the text.
   *  @return text location
   */
  public Point getLocation()
  {
    return new Point(x, y);
  }

  /**
   *  Get the type of game object.
   *  @return game object type
   */
  public abstract String getObjectType();

  /**
   *  Get the shortest vector from this to another game object.
   *
   *  Because of the torus topology it may be shorter going across the borders.
   *  @param obj other object
   *  @return shortest vector from this to the other object
   */
  public Point getDelta(GameObject obj)
  {
    return getDelta(obj.x, obj.y);
  }

  /**
   *  Get the shortest vector from this object to a given point.
   *
   *  Because of the torus topology it may be shorter going across the borders.
   *  @param p point
   *  @return shortest vector from this object's location to the point
   */
  public Point getDelta(Point p)
  {
    return getDelta(p.x, p.y);
  }

  /**
   *  Get the shortest vector from this object to a given point.
   *
   *  Because of the torus topology it may be shorter going across the borders.
   *  @param px point's x coordinate
   *  @param py point's y coordinate
   *  @return shortest vector from this object's location to the point
   */
  public Point getDelta(int px, int py)
  {
    return getTorusDelta(px, py, x, y);
  }

  /**
   *  Get the shortest vector between two points taking care of the torus topology.
   *  @param p1 first point
   *  @param p2 second point
   *  @return delta vector
   */
  public static Point getTorusDelta(Point p1, Point p2)
  {
    return getTorusDelta(p1.x, p1.y, p2.x, p2.y);
  }

  /**
   *  Get the shortest vector between two points taking care of the torus topology.
   *  @param p1x first point's x
   *  @param p1y first point's y
   *  @param p2x second point's x
   *  @param p2y second point's y
   *  @return delta vector
   */
  public static Point getTorusDelta(int p1x, int p1y, int p2x, int p2y)
  {
    int dx = p1x - p2x;
    if (dx < -EXTENT_X/2) {
      dx += EXTENT_X;
    }
    else if (dx >= EXTENT_X/2) {
      dx -= EXTENT_X;
    }
    int dy = p1y - p2y;
    if (dy < -EXTENT_Y/2) {
      dy += EXTENT_Y;
    }
    else if (dy >= EXTENT_Y/2) {
      dy -= EXTENT_Y;
    }
    return new Point(dx, dy);
  }
}
