// ============================================================================
// File:               $File$
//
// Project:            
//
// Purpose:            
//
// Author:             Rammi
//
// Copyright Notice:   (c) 2008  Rammi (rammi@caff.de)
//                     This code is in the public domain.
//                     Use at own risk.
//                     No guarantees given.
//
// Latest change:      $Date$
//
// History:	       $Log$
//=============================================================================
package de.caff.asteroid.analysis;

import de.caff.asteroid.*;
import de.caff.util.KnockOffListener;
import de.caff.util.Worker;

import javax.swing.*;
import java.awt.*;
import java.net.URL;

/**
 *  Applet to display dump files.
 */
public class AnalysisApplet
        extends JApplet
{
  private static final String CARD_STANDARD = "standard";
  private static final String CARD_PROGRESS = "progress";

  public static final String PARAM_FILE = "file";

  private class MinimalInfo
          extends JPanel
          implements FrameListener
  {
    private JLabel fileInfo  = new JLabel();
    private JLabel frameInfo = new JLabel();
    private JLabel keyInfo   = new JLabel();

    MinimalInfo()
    {
      super(new BorderLayout());
      Box box = Box.createHorizontalBox();

      box.add(fileInfo);
      box.add(frameInfo);
      box.add(Box.createHorizontalGlue());
      box.add(keyInfo);
      add(box, BorderLayout.CENTER);
    }
    /**
     * Called each time a frame is received.
     * <p/>
     * <b>ATTENTION:</b> this is called from the communication thread!
     * Implementing classes must be aware of this and take care by synchronization or similar!
     *
     * @param frame the received frame
     */
    public void frameReceived(FrameInfo frame)
    {
      if (frame == null) {
        frameInfo.setText("???");
        keyInfo.setText("???");
      }
      else {
        frameInfo.setText(String.format(" #%d", frame.getIndex()));
        FrameKeyInfo info = timeLine.getCurrentInfo();
        java.util.List<FrameKeyInfo.ButtonInfo> buttons = info.getButtons();
        StringBuilder b = new StringBuilder();
        if (!buttons.isEmpty()) {
          Buttons keys = buttons.get(buttons.size() - 1).getButtons();
          if (keys.isFirePressed()) {
            b.append("<FIRE>");
          }
          if (keys.isHyperspacePressed()) {
            b.append("<HYPER>");
          }
          if (keys.isThrustPressed()) {
            b.append("<THRUST>");
          }
          if (keys.isLeftPressed()) {
            b.append("<LEFT>");
          }
          if (keys.isRightPressed()) {
            b.append("<RIGHT>");
          }
          if (keys.isStartPressed()) {
            b.append("<START>");
          }
        }
        keyInfo.setText(b.length() == 0  ?  "<>"  :  b.toString());
      }
    }

    /**
     *  Refersh dump file information.
     */
    void refreshDumpFile()
    {
      DumpFile dumpFile = timeLine.getDumpFile();
      if (dumpFile == null) {
        fileInfo.setText("???");
        frameReceived(null);
      }
      else {
        StringBuilder b = new StringBuilder(dumpFile.getFilename());
        if (dumpFile.getIpAddress() != null) {
          b.append(" [").append(dumpFile.getIpAddress()).append(":").append(dumpFile.getClientPort());
          if (dumpFile.getPlayerName() != null) {
            b.append("; ").append(dumpFile.getPlayerName());
          }
          b.append("]");
        }
        fileInfo.setText(b.toString());
        frameReceived(timeLine.getCurrentInfo().getFrameInfo());
      }
    }
  }
  private CardLayout cardLayout;
  private JPanel     cardPanel;
  private TimeLine timeLine;
  private ProgressTimeLine progress;
  private MinimalInfo infoLine;


  /**
   * Constructor.
   */
  public AnalysisApplet()
  {
    timeLine = new TimeLine(null);
    progress = new ProgressTimeLine();
    infoLine = new MinimalInfo();
    timeLine.addFrameListener(infoLine);
    infoLine.setBorder(BorderFactory.createEtchedBorder());

    cardLayout = new CardLayout();
    cardPanel = new JPanel(cardLayout);
    cardPanel.add(timeLine, CARD_STANDARD);
    cardPanel.add(progress, CARD_PROGRESS);

    getContentPane().setLayout(new BorderLayout());
    getContentPane().add(cardPanel, BorderLayout.SOUTH);
    getContentPane().add(infoLine, BorderLayout.NORTH);
    getContentPane().add(new FrameKeyInfoDisplay(timeLine, new SimpleVelocityPreparer(), null, true));
  }

  /**
   * Called by the browser or applet viewer to inform
   * this applet that it has been loaded into the system. It is always
   * called before the first time that the <code>start</code> method is
   * called.
   * <p/>
   * A subclass of <code>Applet</code> should override this method if
   * it has initialization to perform. For example, an applet with
   * threads would use the <code>init</code> method to create the
   * threads and the <code>destroy</code> method to kill them.
   * <p/>
   * The implementation of this method provided by the
   * <code>Applet</code> class does nothing.
   *
   * @see java.applet.Applet#destroy()
   * @see java.applet.Applet#start()
   * @see java.applet.Applet#stop()
   */
  @Override
  public void init()
  {
    String file = getParameter(PARAM_FILE);
    if (file != null) {
      showDumpFileDirect(file);
    }
  }

  /**
   *  Set the currently displayed dump file.
   *  @param filename dump file name
   */
  public void showDumpFileDirect(final String filename)
  {
    timeLine.setDumpFile(null);
    infoLine.refreshDumpFile();
    KnockOffListener ko = new KnockOffListener()
    {
      public void knockedOff(Worker worker)
      {
        setCursor(Cursor.getDefaultCursor());
        cardLayout.show(cardPanel, CARD_STANDARD);
        try {
          worker.rethrow();

          DumpLoader loader = (DumpLoader)worker;
          timeLine.setDumpFile(loader.getDumpFile());
          infoLine.refreshDumpFile();
          getAppletContext().showStatus("Loaded "+filename);
        } catch (Throwable x) {
          JOptionPane.showMessageDialog(AnalysisApplet.this,
                                        new Object[] {
                                                "Cannot open '"+filename+"':",
                                                x.getMessage()
                                        },
                                        "Error while reading Dump File",
                                        JOptionPane.ERROR_MESSAGE);
        }
      }
    };
    try {
      URL fileURL = new URL(getDocumentBase(), filename);
      System.out.println("URL: "+fileURL);
      Worker worker = new DumpLoader(fileURL.openStream(),
                                     filename,
                                     ko,
                                     progress);
      setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
      cardLayout.show(cardPanel, CARD_PROGRESS);
      new Thread(worker, "Loader").start();
    } catch (Throwable e) {
      JOptionPane.showMessageDialog(this,
                                    new Object[] {
                                            "Cannot open '"+filename+"':",
                                            e.getMessage()
                                    },
                                    "Error while reading Dump File",
                                    JOptionPane.ERROR_MESSAGE);
    }
  }

  /**
   *  Load a new dump file.
   *  Invoke this method from JavaScript as a workaround to bug #6669818.
   *  @param dumpFile dump file
   */
  public void showDumpFile(final String dumpFile)
  {
    SwingUtilities.invokeLater(new Runnable()
    {
      public void run()
      {
        showDumpFileDirect(dumpFile);
      }
    });
  }
}
