// ============================================================================
// File:               RJFrame.java
//
// Project:            i18n 
//
// Purpose:            JFrame which notifies it's components of locale changes
//
// Author:             Rammi
//-----------------------------------------------------------------------------
// Copyright Notice:   (c) 2002  Rammi (rammi@caff.de)
//                     This code is in the public domain.
//                     Use at own risk.
//                     No guarantees given.
//
// Latest change:      $Date$
//
// History:	       $Log$
//=============================================================================

package de.caff.i18n.swing;


import de.caff.i18n.I18n;
import de.caff.i18n.Localizable;

import javax.swing.*;
import java.awt.*;
import java.util.Locale;

/**
 *  JFrame which notifies it's components of locale changes.
 *  @author <a href="mailto:rammi@caff.de">Rammi</a>
 *  @version $Revision$
 */
public class RJFrame 
  extends    JFrame 
  implements Localizable 
{
  protected String titleRes = null;      // resource for title

  /**
   *
   */
  public RJFrame() {
  }

  /**
   *  @param  titleRes  resource for title
   *  @param  l         locale 
   */
  public RJFrame(String titleRes, Locale l) {
    super(I18n.getString(titleRes, l));
    setLocale(l);
  }


  /**
   * Notifies this component that it now has a parent component.
   * When this method is invoked, the chain of parent components is
   * set up with <code>KeyboardAction</code> event listeners.
   */
  public void addNotify()
  {
    super.addNotify();
    I18n.addLocalizationChangeListener(this);
  }

  /**
   * Overrides <code>JComponent.removeNotify</code> to check if
   * this button is currently set as the default button on the
   * <code>RootPane</code>, and if so, sets the <code>RootPane</code>'s
   * default button to <code>null</code> to ensure the
   * <code>RootPane</code> doesn't hold onto an invalid button reference.
   */
  public void removeNotify()
  {
    I18n.removeLocalizationChangeListener(this);
    super.removeNotify();
  }


  /**
   *  Change the title resource tag.
   *  @param  tag       new resource tag for the name
   */
  public void setTitleTag(String tag) {
    titleRes = tag;
    if (tag != null) {
      setTitle(I18n.getString(tag, getLocale()));
    }
  }

  /**
   *  Change the locale.
   *  @param  l   new locale
   */
  public void setLocale(Locale l) {
    Locale oldLocale = getLocale();
    super.setLocale(l);
    if (oldLocale != null  &&  !oldLocale.equals(l)) {
      if (titleRes != null) {
        setTitle(I18n.getString(titleRes, l));
      }
      validateTree();
    }
  }

  /**
   *  Get the current locale. Don't throws <code>IllegalComponentStateException</code>.
   *  @return current locale or null, if unknown
   */
  public Locale getLocale() {
    try {
      return super.getLocale();
    } catch (IllegalComponentStateException x) {
      return null;
    }
  }

}
